import Link from 'next/link'
import { Card, CardContent, CardFooter } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'

interface BlogPost {
  id: string
  title: string
  excerpt: string
  category: string
  author: string
  date: string
  readTime: string
  views: number
  comments: number
  image: string
}

interface Props {
  post: BlogPost
}

export default function BlogCard({ post }: Props) {
  return (
    <Link href={`/blog/${post.id}`}>
      <Card className="relative glass-dark border border-white/10 card-premium group cursor-pointer h-full min-h-[380px] overflow-hidden hover:border-green-500/50 transition-all duration-300 hover:shadow-2xl hover:shadow-green-500/20 hover:-translate-y-2">
        {/* Gradient Background Effect */}
        <div className="absolute inset-0 bg-gradient-to-br from-green-500/0 via-yellow-500/0 to-emerald-500/0 group-hover:from-green-500/10 group-hover:via-yellow-500/5 group-hover:to-emerald-500/10 transition-all duration-500"></div>
        
        {/* Glow Effect */}
        <div className="absolute -top-24 -right-24 w-48 h-48 bg-green-500/20 rounded-full blur-3xl group-hover:bg-green-500/30 transition-all duration-500"></div>
        
        <CardContent className="p-8 relative z-10">
          <div className="flex flex-col h-full">
            <div className="flex-1">
              {/* Category & Date */}
              <div className="flex items-center space-x-3 mb-5">
                <Badge className="bg-gradient-to-r from-green-500/20 to-yellow-400/20 border border-green-500/40 text-green-300 text-sm px-4 py-1.5 font-semibold shadow-lg shadow-green-500/20 hover:from-green-500/30 hover:to-yellow-400/30 transition-all">
                  {post.category}
                </Badge>
                <span className="text-sm text-foreground/40">•</span>
                <span className="text-sm text-foreground/60 font-medium">{new Date(post.date).toLocaleDateString('tr-TR')}</span>
              </div>

              {/* Title */}
              <h3 className="text-2xl font-black mb-4 group-hover:text-transparent group-hover:bg-gradient-to-r group-hover:from-green-400 group-hover:to-yellow-400 group-hover:bg-clip-text transition-all duration-300 leading-tight">
                {post.title}
              </h3>
              
              {/* Excerpt */}
              <p className="text-base text-foreground/70 line-clamp-4 mb-6 leading-relaxed group-hover:text-foreground/80 transition-colors">{post.excerpt}</p>
            </div>

            {/* Stats Footer */}
            <div className="flex items-center justify-between text-sm pt-5 border-t border-white/10 group-hover:border-green-500/30 transition-colors">
              <div className="flex items-center space-x-4 text-foreground/60">
                <div className="flex items-center space-x-2 hover:text-green-400 transition-colors">
                  <div className="h-1.5 w-1.5 rounded-full bg-green-400"></div>
                  <span className="font-medium">{post.views} görüntülenme</span>
                </div>
                <div className="flex items-center space-x-2 hover:text-yellow-400 transition-colors">
                  <div className="h-1.5 w-1.5 rounded-full bg-yellow-400"></div>
                  <span className="font-medium">{post.comments} yorum</span>
                </div>
              </div>
              <div className="bg-gradient-to-r from-green-500/10 to-yellow-400/10 px-3 py-1.5 rounded-lg border border-green-500/20">
                <span className="font-bold text-green-300">{post.readTime}</span>
              </div>
            </div>
          </div>
        </CardContent>
        
        {/* Bottom Accent Line */}
        <div className="absolute bottom-0 left-0 right-0 h-1 bg-gradient-to-r from-green-500/0 via-green-500/50 to-green-500/0 group-hover:via-green-500 transition-all duration-500"></div>
      </Card>
    </Link>
  )
}

