import { NextRequest, NextResponse } from 'next/server'
import OpenAI from 'openai'

const FOOTBALL_API_KEY = '303a19f14f12a119e136c61bdf8262ea'

// Football API'den maçı ve oranları ara
async function findMatchWithOdds(homeTeam: string, awayTeam: string) {
  try {
    // Bugünün maçlarını çek
    const todayIstanbul = new Intl.DateTimeFormat('en-CA', {
      timeZone: 'Europe/Istanbul',
      year: 'numeric',
      month: '2-digit',
      day: '2-digit'
    }).format(new Date())

    const url = `https://v3.football.api-sports.io/fixtures?date=${todayIstanbul}&timezone=Europe/Istanbul`
    
    const response = await fetch(url, {
      headers: {
        'x-rapidapi-key': FOOTBALL_API_KEY,
        'x-rapidapi-host': 'v3.football.api-sports.io',
        'x-apisports-key': FOOTBALL_API_KEY
      }
    })

    if (!response.ok) return null

    const data = await response.json()
    const matches = data.response || []

    // Takım isimlerine göre maçı bul
    const match = matches.find((m: any) => {
      const home = m.teams.home.name.toLowerCase()
      const away = m.teams.away.name.toLowerCase()
      const searchHome = homeTeam.toLowerCase()
      const searchAway = awayTeam.toLowerCase()
      
      return (home.includes(searchHome) || searchHome.includes(home)) &&
             (away.includes(searchAway) || searchAway.includes(away))
    })

    if (!match) return null

    // Oranları çek
    const oddsUrl = `https://v3.football.api-sports.io/odds?fixture=${match.fixture.id}`
    const oddsResponse = await fetch(oddsUrl, {
      headers: {
        'x-rapidapi-key': FOOTBALL_API_KEY,
        'x-rapidapi-host': 'v3.football.api-sports.io',
        'x-apisports-key': FOOTBALL_API_KEY
      }
    })

    if (oddsResponse.ok) {
      const oddsData = await oddsResponse.json()
      match.odds = oddsData.response?.[0]?.bookmakers?.[0]?.bets || []
    }

    return match
  } catch (error) {
    console.error('Football API hatası:', error)
    return null
  }
}

export async function POST(request: NextRequest) {
  try {
    // Herkes AI tahmin alabilir - auth kontrolü kaldırıldı
    const body = await request.json()
    const { homeTeam, awayTeam, league, matchDate } = body

    if (!homeTeam || !awayTeam) {
      return NextResponse.json(
        { error: 'Takım bilgileri eksik' },
        { status: 400 }
      )
    }

    // API Key kontrolü
    if (!process.env.OPENAI_API_KEY) {
      return NextResponse.json(
        { error: 'OpenAI API key tanımlanmamış. Lütfen .env dosyasına OPENAI_API_KEY ekleyin.' },
        { status: 500 }
      )
    }

    // Gerçek maçı ve oranları bul
    const realMatch = await findMatchWithOdds(homeTeam, awayTeam)
    
    let realOdds = ''
    if (realMatch) {
      const homeWinOdds = realMatch.odds?.find((b: any) => b.name === 'Match Winner')?.values?.find((v: any) => v.value === 'Home')?.odd
      const drawOdds = realMatch.odds?.find((b: any) => b.name === 'Match Winner')?.values?.find((v: any) => v.value === 'Draw')?.odd
      const awayWinOdds = realMatch.odds?.find((b: any) => b.name === 'Match Winner')?.values?.find((v: any) => v.value === 'Away')?.odd
      const over25 = realMatch.odds?.find((b: any) => b.name === 'Goals Over/Under')?.values?.find((v: any) => v.value === 'Over 2.5')?.odd
      const under25 = realMatch.odds?.find((b: any) => b.name === 'Goals Over/Under')?.values?.find((v: any) => v.value === 'Under 2.5')?.odd
      const bttsYes = realMatch.odds?.find((b: any) => b.name === 'Both Teams Score')?.values?.find((v: any) => v.value === 'Yes')?.odd
      const bttsNo = realMatch.odds?.find((b: any) => b.name === 'Both Teams Score')?.values?.find((v: any) => v.value === 'No')?.odd

      realOdds = `\n\nGERÇEK ORANLAR (Bahis sitelerinden):\n- MS1 (Ev Sahibi Kazanır): ${homeWinOdds || 'Yok'}\n- Beraberlik: ${drawOdds || 'Yok'}\n- MS2 (Deplasman Kazanır): ${awayWinOdds || 'Yok'}\n- 2.5 Üst: ${over25 || 'Yok'}\n- 2.5 Alt: ${under25 || 'Yok'}\n- Karşılıklı Gol Var: ${bttsYes || 'Yok'}\n- Karşılıklı Gol Yok: ${bttsNo || 'Yok'}`
    }

    const openai = new OpenAI({
      apiKey: process.env.OPENAI_API_KEY,
    })

    // ChatGPT'ye gönderilecek prompt
    const prompt = `Sen profesyonel bir iddaa analisti ve tahmin uzmanısın. Aşağıdaki maç için detaylı analiz ve tahmin yap:

Maç: ${homeTeam} vs ${awayTeam}
Lig: ${league || 'Bilinmiyor'}
Tarih: ${matchDate || 'Yakında'}${realOdds}

${realOdds ? 'ÇOK ÖNEMLİ: Yukarıda verilen GERÇEK oranları kullanmalısın! Kendi oran uydurma!' : ''}

Lütfen şu formatta yanıt ver (JSON):
{
  "analysis": "Detaylı maç analizi (2-3 cümle)",
  "predictions": [
    {
      "type": "Maç Sonucu",
      "prediction": "MS1 veya Beraberlik veya MS2",
      "confidence": "Yüksek/Orta/Düşük",
      "odds": "${realOdds ? 'Yukarıdaki gerçek oranı kullan' : 'Tahmini oran'}"
    },
    {
      "type": "Alt/Üst 2.5",
      "prediction": "Üst veya Alt",
      "confidence": "Yüksek/Orta/Düşük",
      "odds": "${realOdds ? 'Yukarıdaki gerçek oranı kullan' : 'Tahmini oran'}"
    },
    {
      "type": "Karşılıklı Gol",
      "prediction": "Var veya Yok",
      "confidence": "Yüksek/Orta/Düşük",
      "odds": "${realOdds ? 'Yukarıdaki gerçek oranı kullan' : 'Tahmini oran'}"
    }
  ],
  "bankoTip": "En güvenilir tahmin hangisi (type belirt)",
  "riskLevel": "Düşük/Orta/Yüksek"
}

${realOdds ? 'UYARI: Odds değerlerini yukarıdaki GERÇEK oranlardan al! MS1 için Home odd, Beraberlik için Draw odd, MS2 için Away odd, 2.5 Üst için Over 2.5 odd kullan!' : 'Not: Gerçekçi oranlar ver. Takım formlarını düşünerek analiz yap.'}`

    const completion = await openai.chat.completions.create({
      model: 'gpt-4o-mini',
      messages: [
        {
          role: 'system',
          content: 'Sen profesyonel bir iddaa analisti ve tahmin uzmanısın. Takım formlarını, istatistikleri ve maç analizlerini yaparak doğru tahminler veriyorsun. ÖNEMLİ: Eğer gerçek oranlar verilmişse, AYNEN o oranları kullanmalısın, kendi oran uydurmamalısın! Yanıtların her zaman JSON formatında olmalı.'
        },
        {
          role: 'user',
          content: prompt
        }
      ],
      temperature: 0.6,
      max_tokens: 1000,
      response_format: { type: "json_object" }
    })

    const aiResponse = completion.choices[0].message.content
    if (!aiResponse) {
      throw new Error('AI yanıt vermedi')
    }

    const prediction = JSON.parse(aiResponse)

    return NextResponse.json({
      success: true,
      prediction,
      usage: completion.usage,
    })
  } catch (error: any) {
    console.error('AI tahmin hatası:', error)
    return NextResponse.json(
      { 
        error: error?.message || 'AI tahmin oluşturulamadı',
        details: error?.response?.data || null
      },
      { status: 500 }
    )
  }
}

