import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { prisma } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const session = await auth()
    
    if (!session?.user || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Yetkisiz erişim' }, { status: 403 })
    }

    const { searchParams } = new URL(request.url)
    const page = parseInt(searchParams.get('page') || '1')
    const limit = parseInt(searchParams.get('limit') || '20')
    const search = searchParams.get('search')
    
    const skip = (page - 1) * limit

    // Build where clause
    const where: any = {}
    
    if (search) {
      where.OR = [
        { name: { contains: search } },
        { username: { contains: search } }
      ]
    }

    // Get tipsters (all users) sorted by win rate and total coupons
    const [tipsters, total] = await Promise.all([
      prisma.user.findMany({
        where,
        skip,
        take: limit,
        orderBy: [
          { winRate: 'desc' }, // En başarılı olanlar üstte
          { totalCoupons: 'desc' } // Sonra kupon sayısına göre
        ],
        include: {
          _count: {
            select: {
              coupons: true,
              followers: true,
              following: true
            }
          }
        }
      }),
      prisma.user.count({ where })
    ])

    // Get stats
    const totalTipsters = await prisma.user.count()
    
    const topTipsters = await prisma.user.count({
      where: { 
        winRate: { gte: 70 }
      }
    })

    return NextResponse.json({
      tipsters: tipsters.map(user => ({
        ...user,
        password: undefined // Don't send password
      })),
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit)
      },
      stats: {
        total: totalTipsters,
        topTipsters: topTipsters
      }
    })
  } catch (error) {
    console.error('Error fetching tipsters:', error)
    return NextResponse.json({ error: 'Tahminçiler yüklenemedi' }, { status: 500 })
  }
}

