import { NextRequest, NextResponse } from 'next/server'
import { auth } from '@/auth'
import { prisma } from '@/lib/db'

export async function GET(request: NextRequest) {
  try {
    const session = await auth()
    
    if (!session?.user || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Yetkisiz erişim' }, { status: 403 })
    }

    // Get all stats in parallel
    const [
      totalUsers,
      totalCoupons,
      totalComments,
      wonCoupons,
      lostCoupons,
      pendingCoupons,
      topCoupons,
      recentUsers
    ] = await Promise.all([
      // Total counts
      prisma.user.count(),
      prisma.coupon.count(),
      prisma.comment.count(),
      
      // Coupon status counts
      prisma.coupon.count({ where: { status: 'WON' } }),
      prisma.coupon.count({ where: { status: 'LOST' } }),
      prisma.coupon.count({ where: { status: 'PENDING' } }),
      
      // Top coupons by views
      prisma.coupon.findMany({
        take: 5,
        orderBy: { viewCount: 'desc' },
        include: {
          user: {
            select: { username: true, name: true }
          },
          _count: {
            select: {
              likes: true,
              comments: true
            }
          }
        }
      }),
      
      // Recent users (last 7 days)
      prisma.user.count({
        where: {
          createdAt: {
            gte: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000)
          }
        }
      })
    ])

    // Calculate success rate
    const completedCoupons = wonCoupons + lostCoupons
    const successRate = completedCoupons > 0 ? ((wonCoupons / completedCoupons) * 100).toFixed(1) : 0

    return NextResponse.json({
      stats: {
        totalUsers,
        totalCoupons,
        totalComments,
        wonCoupons,
        lostCoupons,
        pendingCoupons,
        recentUsers,
        successRate
      },
      topCoupons: topCoupons.map(coupon => ({
        id: coupon.id,
        title: coupon.title,
        views: coupon.viewCount,
        likes: coupon._count.likes,
        comments: coupon._count.comments,
        user: coupon.user?.username || 'Anonim',
        status: coupon.status
      }))
    })
  } catch (error) {
    console.error('Error fetching stats:', error)
    return NextResponse.json({ error: 'İstatistikler yüklenemedi' }, { status: 500 })
  }
}





