'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { 
  Search,
  SlidersHorizontal,
  Trophy,
  Eye,
  Edit,
  Ban,
  Star,
  TrendingUp,
  TrendingDown,
  Users,
  Target,
  Award,
  Crown,
  MoreVertical,
  Download,
  CheckCircle,
  XCircle,
  Zap,
  Calendar,
  FileText,
  Trash2,
  X,
  AlertCircle,
  UserPlus,
  Activity
} from 'lucide-react'

export default function AdminTahmincilarPage() {
  const [searchQuery, setSearchQuery] = useState('')
  const [selectedTipsters, setSelectedTipsters] = useState<string[]>([])
  const [loading, setLoading] = useState(true)
  const [page, setPage] = useState(1)
  const [limit] = useState(20)

  // Modal states
  const [showDetailModal, setShowDetailModal] = useState(false)
  const [selectedTipster, setSelectedTipster] = useState<any>(null)

  // Real data states
  const [stats, setStats] = useState([
    { label: 'Toplam Kullanıcı', value: '0', icon: Users, color: 'blue' },
    { label: 'Başarılı Kullanıcı (%70+)', value: '0', icon: Trophy, color: 'yellow' }
  ])

  const [tipsters, setTipsters] = useState<any[]>([])
  const [total, setTotal] = useState(0)

  // Load tipsters
  const loadTipsters = async () => {
    setLoading(true)
    try {
      const params = new URLSearchParams()
      params.set('page', String(page))
      params.set('limit', String(limit))
      if (searchQuery) params.set('search', searchQuery)

      const res = await fetch(`/api/admin/tipsters?${params.toString()}`, { cache: 'no-store' })
      const data = await res.json()
      
      if (data.tipsters) {
        setTipsters(data.tipsters)
        setTotal(data.pagination?.total || 0)
        
        // Update stats
        if (data.stats) {
          setStats([
            { label: 'Toplam Kullanıcı', value: String(data.stats.total || 0), icon: Users, color: 'blue' },
            { label: 'Başarılı Kullanıcı (%70+)', value: String(data.stats.topTipsters || 0), icon: Trophy, color: 'yellow' }
          ])
        }
      }
    } catch (error) {
      console.error('Error loading tipsters:', error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    loadTipsters()
  }, [page])

  // Search with debounce
  useEffect(() => {
    const timer = setTimeout(() => {
      if (page === 1) {
        loadTipsters()
      } else {
        setPage(1)
      }
    }, 500)
    return () => clearTimeout(timer)
  }, [searchQuery])

  // Calculate tier based on win rate and total coupons
  const getTierBadge = (winRate: number, totalCoupons: number) => {
    if (totalCoupons >= 50 && winRate >= 80) {
      return <Badge className="bg-gradient-to-r from-yellow-500 to-orange-500 border-0"><Crown className="h-3 w-3 mr-1" />Efsane</Badge>
    } else if (totalCoupons >= 30 && winRate >= 75) {
      return <Badge className="bg-gradient-to-r from-purple-500 to-pink-500 border-0"><Star className="h-3 w-3 mr-1" />Master</Badge>
    } else if (totalCoupons >= 20 && winRate >= 70) {
      return <Badge className="bg-gradient-to-r from-blue-500 to-cyan-500 border-0"><Trophy className="h-3 w-3 mr-1" />Uzman</Badge>
    } else if (totalCoupons >= 10 && winRate >= 60) {
      return <Badge className="bg-gradient-to-r from-green-500 to-emerald-500 border-0"><Zap className="h-3 w-3 mr-1" />İleri</Badge>
    } else {
      return <Badge className="bg-gray-500/20 border-gray-500/30 text-gray-400">Başlangıç</Badge>
    }
  }

  const handleSelectTipster = (id: string) => {
    setSelectedTipsters(prev => 
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    )
  }

  const handleSelectAll = () => {
    if (selectedTipsters.length === tipsters.length) {
      setSelectedTipsters([])
    } else {
      setSelectedTipsters(tipsters.map(t => t.id))
    }
  }

  const openDetailModal = (tipster: any) => {
    setSelectedTipster(tipster)
    setShowDetailModal(true)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('tr-TR')
  }

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold gradient-text mb-2">Tahminci Sıralaması</h1>
          <p className="text-foreground/60">Kullanıcıları başarı oranına göre sıralı listele</p>
        </div>
        <div className="flex items-center space-x-3">
          <Button variant="outline" className="glass border-white/10" onClick={() => loadTipsters()}>
            <Activity className="h-4 w-4 mr-2" />
            Yenile
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-2 gap-4">
        {stats.map((stat, index) => {
          const Icon = stat.icon
          const colorClasses = {
            blue: { bg: 'bg-blue-500/10 border-blue-500/30', text: 'text-blue-400' },
            yellow: { bg: 'bg-yellow-500/10 border-yellow-500/30', text: 'text-yellow-400' }
          }
          const colors = colorClasses[stat.color as keyof typeof colorClasses] || colorClasses.blue
          return (
            <Card key={index} className="glass-dark border-white/5 hover:border-white/10 transition-all group cursor-pointer">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className={`h-12 w-12 rounded-xl ${colors.bg} border flex items-center justify-center group-hover:scale-110 transition-transform`}>
                    <Icon className={`h-6 w-6 ${colors.text}`} />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">{stat.label}</p>
                <p className="text-3xl font-bold">{stat.value}</p>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {/* Filters & Search */}
      <Card className="glass-dark border-white/5">
        <CardContent className="p-6">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-foreground/40" />
              <Input
                type="search"
                placeholder="Tahminci adı veya kullanıcı adı ara..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="glass-dark border-white/10 pl-10"
              />
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Tipsters Table */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center space-x-2">
            <Trophy className="h-5 w-5 text-blue-400" />
            <span>Tahminci Sıralaması (Başarı Oranına Göre)</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-0">
          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block h-8 w-8 animate-spin rounded-full border-4 border-solid border-orange-500 border-r-transparent"></div>
              <p className="mt-4 text-foreground/60">Tahminçiler yükleniyor...</p>
            </div>
          ) : tipsters.length === 0 ? (
            <div className="p-12 text-center">
              <Trophy className="h-16 w-16 mx-auto text-foreground/20 mb-4" />
              <p className="text-foreground/60">Kullanıcı bulunamadı</p>
              <p className="text-sm text-foreground/40 mt-2">Arama kriterlerinizi değiştirmeyi deneyin</p>
            </div>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="border-b border-white/5 bg-white/5">
                  <tr>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">Sıra</th>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">Tahminci</th>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">Seviye</th>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">İstatistikler</th>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">Başarı Oranı</th>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">Takipçi</th>
                    <th className="text-left p-4 text-sm font-semibold text-foreground/70">Rol</th>
                    <th className="text-right p-4 text-sm font-semibold text-foreground/70">İşlemler</th>
                  </tr>
                </thead>
                <tbody>
                  {tipsters.map((tipster, index) => (
                    <tr 
                      key={tipster.id}
                      className="border-b border-white/5 hover:bg-white/5 transition-colors"
                    >
                      <td className="p-4">
                        <div className="flex items-center space-x-2">
                          {index < 3 && (
                            <div className={`h-8 w-8 rounded-full flex items-center justify-center font-bold text-white ${
                              index === 0 ? 'bg-gradient-to-br from-yellow-400 to-yellow-600' :
                              index === 1 ? 'bg-gradient-to-br from-gray-300 to-gray-500' :
                              'bg-gradient-to-br from-orange-400 to-orange-600'
                            }`}>
                              {index + 1}
                            </div>
                          )}
                          {index >= 3 && (
                            <span className="text-foreground/60 font-semibold">{index + 1}</span>
                          )}
                        </div>
                      </td>
                      <td className="p-4">
                        <div className="flex items-center space-x-3">
                          <div className="h-10 w-10 rounded-full bg-gradient-to-br from-purple-500 to-pink-500 flex items-center justify-center text-white font-bold text-sm">
                            {(tipster.name || tipster.username || 'U').charAt(0).toUpperCase()}
                          </div>
                          <div>
                            <p className="font-semibold text-sm">{tipster.name || 'İsimsiz'}</p>
                            <p className="text-xs text-foreground/60">@{tipster.username}</p>
                          </div>
                        </div>
                      </td>
                      <td className="p-4">
                        {getTierBadge(tipster.winRate || 0, tipster.totalCoupons || 0)}
                      </td>
                      <td className="p-4">
                        <div className="flex flex-col space-y-1">
                          <span className="text-xs text-blue-400 font-semibold">{tipster.totalCoupons} kupon</span>
                          <span className="text-xs text-green-400">{tipster.wonCoupons} kazanan</span>
                          <span className="text-xs text-orange-400">₺{(tipster.totalProfit || 0).toLocaleString('tr-TR')} kâr</span>
                        </div>
                      </td>
                      <td className="p-4">
                        <div className="flex items-center space-x-2">
                          <div className="flex-1 max-w-[100px]">
                            <div className="h-2 bg-white/10 rounded-full overflow-hidden">
                              <div 
                                className="h-full bg-gradient-to-r from-green-500 to-emerald-500"
                                style={{ width: `${Math.min(tipster.winRate || 0, 100)}%` }}
                              />
                            </div>
                          </div>
                          <span className="text-sm font-semibold text-green-400">{(tipster.winRate || 0).toFixed(1)}%</span>
                        </div>
                      </td>
                      <td className="p-4">
                        <div className="flex items-center space-x-1">
                          <Users className="h-4 w-4 text-foreground/40" />
                          <span className="text-sm font-semibold">{tipster._count?.followers || 0}</span>
                        </div>
                      </td>
                      <td className="p-4">
                        {tipster.role === 'ADMIN' ? (
                          <Badge className="bg-red-500/20 border-red-500/30 text-red-400">
                            <Crown className="h-3 w-3 mr-1" />
                            Admin
                          </Badge>
                        ) : tipster.role === 'MODERATOR' ? (
                          <Badge className="bg-purple-500/20 border-purple-500/30 text-purple-400">
                            <Star className="h-3 w-3 mr-1" />
                            Moderatör
                          </Badge>
                        ) : (
                          <Badge className="bg-blue-500/20 border-blue-500/30 text-blue-400">
                            Kullanıcı
                          </Badge>
                        )}
                      </td>
                      <td className="p-4">
                        <div className="flex items-center justify-end space-x-2">
                          <Button 
                            size="sm" 
                            variant="ghost" 
                            className="h-8 w-8 p-0 hover:bg-blue-500/10 hover:text-blue-400"
                            onClick={() => openDetailModal(tipster)}
                          >
                            <Eye className="h-4 w-4" />
                          </Button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}

          <div className="p-4 border-t border-white/5">
            <p className="text-sm text-foreground/60">
              Toplam <strong className="text-orange-400">{total}</strong> tahminciden <strong className="text-blue-400">{tipsters.length}</strong> tanesi gösteriliyor
            </p>
          </div>
        </CardContent>
      </Card>

      {/* Detail Modal */}
      {showDetailModal && selectedTipster && (
        <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <Card className="glass-dark border-white/10 w-full max-w-2xl">
            <CardHeader className="border-b border-white/5 bg-gradient-to-r from-purple-500/10 to-pink-500/10">
              <CardTitle className="flex items-center justify-between">
                <div className="flex items-center space-x-3">
                  <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-purple-500 to-pink-500 flex items-center justify-center">
                    <Trophy className="h-6 w-6 text-white" />
                  </div>
                  <div>
                    <h2 className="text-xl font-bold gradient-text">Tahminci Detayları</h2>
                    <p className="text-xs text-foreground/60 font-normal">Performans ve istatistikler</p>
                  </div>
                </div>
                <Button variant="ghost" size="sm" onClick={() => setShowDetailModal(false)}>
                  <X className="h-5 w-5" />
                </Button>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-6 space-y-6">
              <div className="flex items-center space-x-4">
                <div className="h-20 w-20 rounded-full bg-gradient-to-br from-purple-500 to-pink-500 flex items-center justify-center text-white font-bold text-3xl">
                  {(selectedTipster.name || selectedTipster.username || 'U').charAt(0).toUpperCase()}
                </div>
                <div>
                  <h3 className="text-2xl font-bold">{selectedTipster.name || 'İsimsiz'}</h3>
                  <p className="text-foreground/60">@{selectedTipster.username}</p>
                  <div className="flex items-center space-x-2 mt-2">
                    {getTierBadge(selectedTipster.winRate || 0, selectedTipster.totalCoupons || 0)}
                  </div>
                </div>
              </div>

              {selectedTipster.bio && (
                <div className="glass-dark p-4 rounded-lg border border-white/10">
                  <p className="text-sm text-foreground/80">{selectedTipster.bio}</p>
                </div>
              )}

              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div className="glass-dark p-4 rounded-lg text-center border border-blue-500/20">
                  <p className="text-2xl font-bold text-blue-400">{selectedTipster.totalCoupons || 0}</p>
                  <p className="text-xs text-foreground/60">Toplam Kupon</p>
                </div>
                <div className="glass-dark p-4 rounded-lg text-center border border-green-500/20">
                  <p className="text-2xl font-bold text-green-400">{(selectedTipster.winRate || 0).toFixed(1)}%</p>
                  <p className="text-xs text-foreground/60">Başarı Oranı</p>
                </div>
                <div className="glass-dark p-4 rounded-lg text-center border border-purple-500/20">
                  <p className="text-2xl font-bold text-purple-400">{selectedTipster._count?.followers || 0}</p>
                  <p className="text-xs text-foreground/60">Takipçi</p>
                </div>
                <div className="glass-dark p-4 rounded-lg text-center border border-orange-500/20">
                  <p className="text-2xl font-bold text-orange-400">₺{(selectedTipster.totalProfit || 0).toLocaleString('tr-TR')}</p>
                  <p className="text-xs text-foreground/60">Toplam Kâr</p>
                </div>
              </div>

              <div className="grid grid-cols-3 gap-4">
                <div className="glass-dark p-4 rounded-lg border border-white/10">
                  <p className="text-sm text-foreground/60 mb-2">Kazanan Kuponlar</p>
                  <p className="text-xl font-bold text-green-400">{selectedTipster.wonCoupons || 0}</p>
                </div>
                <div className="glass-dark p-4 rounded-lg border border-white/10">
                  <p className="text-sm text-foreground/60 mb-2">Kaybeden Kuponlar</p>
                  <p className="text-xl font-bold text-red-400">{selectedTipster.lostCoupons || 0}</p>
                </div>
                <div className="glass-dark p-4 rounded-lg border border-white/10">
                  <p className="text-sm text-foreground/60 mb-2">Bekleyen Kuponlar</p>
                  <p className="text-xl font-bold text-yellow-400">{selectedTipster.pendingCoupons || 0}</p>
                </div>
              </div>

              <div className="space-y-3 pt-4 border-t border-white/5">
                <div className="flex justify-between">
                  <span className="text-foreground/60">Kayıt Tarihi:</span>
                  <span className="font-semibold">{formatDate(selectedTipster.createdAt)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-foreground/60">Email:</span>
                  <span className="font-semibold">{selectedTipster.email}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-foreground/60">Rol:</span>
                  <Badge className={
                    selectedTipster.role === 'ADMIN' ? 'bg-red-500/20 border-red-500/30 text-red-400' :
                    selectedTipster.role === 'MODERATOR' ? 'bg-purple-500/20 border-purple-500/30 text-purple-400' :
                    'bg-blue-500/20 border-blue-500/30 text-blue-400'
                  }>
                    {selectedTipster.role}
                  </Badge>
                </div>
              </div>

              <Button 
                className="w-full glass border-white/10 h-12" 
                onClick={() => setShowDetailModal(false)}
              >
                Kapat
              </Button>
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  )
}