'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { 
  FileText,
  Users,
  Trophy,
  MessageSquare,
  Activity,
  BarChart3,
  CheckCircle,
  XCircle,
  Clock,
  Download,
  TrendingUp
} from 'lucide-react'

export default function AdminRaporlarPage() {
  const [loading, setLoading] = useState(true)
  const [stats, setStats] = useState<any>(null)

  const loadStats = async () => {
    setLoading(true)
    try {
      const res = await fetch('/api/admin/stats', { cache: 'no-store' })
      const data = await res.json()
      
      if (data.stats) {
        setStats(data.stats)
      }
    } catch (error) {
      console.error('Error loading stats:', error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    loadStats()
  }, [])

  const downloadReport = (type: string) => {
    if (!stats) return
    
    let content = ''
    const date = new Date().toLocaleString('tr-TR')
    
    if (type === 'summary') {
      content = `PLATFORM ÖZET RAPORU\nOluşturma Tarihi: ${date}\n\n`
      content += `=== GENEL İSTATİSTİKLER ===\n`
      content += `Toplam Kullanıcı: ${stats.totalUsers}\n`
      content += `Toplam Kupon: ${stats.totalCoupons}\n`
      content += `Toplam Yorum: ${stats.totalComments}\n`
      content += `Başarı Oranı: ${stats.successRate}%\n\n`
      content += `=== KUPON DAĞILIMI ===\n`
      content += `Kazanan: ${stats.wonCoupons}\n`
      content += `Kaybeden: ${stats.lostCoupons}\n`
      content += `Bekleyen: ${stats.pendingCoupons}\n\n`
      content += `Son 7 Günde Yeni Kullanıcı: ${stats.recentUsers}\n`
    }

    const blob = new Blob([content], { type: 'text/plain;charset=utf-8' })
    const url = window.URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url
    a.download = `platform-rapor-${Date.now()}.txt`
    document.body.appendChild(a)
    a.click()
    document.body.removeChild(a)
    window.URL.revokeObjectURL(url)
  }

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold gradient-text mb-2">Rapor Merkezi</h1>
          <p className="text-foreground/60">Platform istatistiklerini görüntüle ve raporları indir</p>
        </div>
        <Button 
          variant="outline" 
          className="glass border-white/10"
          onClick={loadStats}
        >
          <Activity className="h-4 w-4 mr-2" />
          Yenile
        </Button>
      </div>

      {loading ? (
        <div className="p-12 text-center">
          <div className="inline-block h-8 w-8 animate-spin rounded-full border-4 border-solid border-orange-500 border-r-transparent"></div>
          <p className="mt-4 text-foreground/60">İstatistikler yükleniyor...</p>
        </div>
      ) : stats ? (
        <>
          {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <Card className="glass-dark border-white/5">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-blue-500/10 border border-blue-500/30 flex items-center justify-center">
                    <Users className="h-6 w-6 text-blue-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Toplam Kullanıcı</p>
                <p className="text-3xl font-bold">{stats.totalUsers}</p>
              </CardContent>
            </Card>

            <Card className="glass-dark border-white/5">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-purple-500/10 border border-purple-500/30 flex items-center justify-center">
                    <FileText className="h-6 w-6 text-purple-400" />
                  </div>
      </div>
                <p className="text-foreground/60 text-sm mb-1">Toplam Kupon</p>
                <p className="text-3xl font-bold">{stats.totalCoupons}</p>
          </CardContent>
        </Card>

            <Card className="glass-dark border-white/5">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-green-500/10 border border-green-500/30 flex items-center justify-center">
                    <MessageSquare className="h-6 w-6 text-green-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Toplam Yorum</p>
                <p className="text-3xl font-bold">{stats.totalComments}</p>
          </CardContent>
        </Card>

            <Card className="glass-dark border-white/5">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-orange-500/10 border border-orange-500/30 flex items-center justify-center">
                    <Trophy className="h-6 w-6 text-orange-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Başarı Oranı</p>
                <p className="text-3xl font-bold">{stats.successRate}%</p>
          </CardContent>
        </Card>
      </div>

          {/* Detailed Stats */}
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
            <Card className="glass-dark border-green-500/20">
              <CardHeader className="border-b border-white/5">
                <CardTitle className="flex items-center space-x-2">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-green-400">Kazanan Kuponlar</span>
                </CardTitle>
              </CardHeader>
        <CardContent className="p-6">
                <p className="text-4xl font-bold text-green-400 mb-2">{stats.wonCoupons}</p>
                <div className="h-2 bg-white/5 rounded-full overflow-hidden">
                  <div 
                    className="h-full bg-gradient-to-r from-green-500 to-emerald-500"
                    style={{ width: `${stats.totalCoupons > 0 ? (stats.wonCoupons / stats.totalCoupons * 100) : 0}%` }}
              />
            </div>
                <p className="text-xs text-foreground/60 mt-2">
                  Toplam kuponların {stats.totalCoupons > 0 ? ((stats.wonCoupons / stats.totalCoupons * 100).toFixed(1)) : 0}%
                </p>
        </CardContent>
      </Card>

            <Card className="glass-dark border-red-500/20">
        <CardHeader className="border-b border-white/5">
                <CardTitle className="flex items-center space-x-2">
                  <XCircle className="h-5 w-5 text-red-400" />
                  <span className="text-red-400">Kaybeden Kuponlar</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
                <p className="text-4xl font-bold text-red-400 mb-2">{stats.lostCoupons}</p>
                <div className="h-2 bg-white/5 rounded-full overflow-hidden">
                  <div 
                    className="h-full bg-gradient-to-r from-red-500 to-rose-500"
                    style={{ width: `${stats.totalCoupons > 0 ? (stats.lostCoupons / stats.totalCoupons * 100) : 0}%` }}
                  />
                </div>
                <p className="text-xs text-foreground/60 mt-2">
                  Toplam kuponların {stats.totalCoupons > 0 ? ((stats.lostCoupons / stats.totalCoupons * 100).toFixed(1)) : 0}%
                </p>
        </CardContent>
      </Card>

            <Card className="glass-dark border-yellow-500/20">
            <CardHeader className="border-b border-white/5">
                <CardTitle className="flex items-center space-x-2">
                  <Clock className="h-5 w-5 text-yellow-400" />
                  <span className="text-yellow-400">Bekleyen Kuponlar</span>
              </CardTitle>
            </CardHeader>
              <CardContent className="p-6">
                <p className="text-4xl font-bold text-yellow-400 mb-2">{stats.pendingCoupons}</p>
                <div className="h-2 bg-white/5 rounded-full overflow-hidden">
                  <div 
                    className="h-full bg-gradient-to-r from-yellow-500 to-amber-500"
                    style={{ width: `${stats.totalCoupons > 0 ? (stats.pendingCoupons / stats.totalCoupons * 100) : 0}%` }}
                  />
                </div>
                <p className="text-xs text-foreground/60 mt-2">
                  Toplam kuponların {stats.totalCoupons > 0 ? ((stats.pendingCoupons / stats.totalCoupons * 100).toFixed(1)) : 0}%
                </p>
            </CardContent>
          </Card>
        </div>

          {/* Download Report */}
          <Card className="glass-dark border-white/5">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="flex items-center space-x-2">
                <BarChart3 className="h-5 w-5 text-blue-400" />
                <span>Rapor İndir</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Button 
                  onClick={() => downloadReport('summary')}
                  className="h-20 bg-gradient-to-r from-blue-500 to-cyan-500 hover:from-blue-600 hover:to-cyan-600 text-left justify-start"
                >
                  <div className="flex items-center space-x-4">
                    <div className="h-12 w-12 rounded-xl bg-white/20 flex items-center justify-center">
                      <Download className="h-6 w-6" />
                    </div>
                    <div>
                      <p className="font-bold text-lg">Özet Rapor</p>
                      <p className="text-xs opacity-80">Tüm istatistikler (.txt)</p>
                    </div>
                  </div>
                </Button>

                <Card className="glass-dark border-white/10 p-6">
                  <div className="flex items-center space-x-4">
                    <div className="h-12 w-12 rounded-xl bg-green-500/20 border border-green-500/30 flex items-center justify-center">
                      <TrendingUp className="h-6 w-6 text-green-400" />
                    </div>
                    <div>
                      <p className="text-sm text-foreground/60">Son 7 Günde</p>
                      <p className="text-2xl font-bold text-green-400">+{stats.recentUsers}</p>
                      <p className="text-xs text-foreground/60">Yeni Kullanıcı</p>
                    </div>
                  </div>
                </Card>
              </div>

              <div className="mt-6 p-4 glass-dark rounded-lg border border-blue-500/20">
                <h4 className="font-semibold mb-3 flex items-center space-x-2">
                  <FileText className="h-4 w-4 text-blue-400" />
                  <span>Rapor İçeriği</span>
                </h4>
                <ul className="space-y-2 text-sm text-foreground/70">
                  <li>✓ Toplam kullanıcı, kupon ve yorum sayıları</li>
                  <li>✓ Platform başarı oranı</li>
                  <li>✓ Kupon durum dağılımı (Kazanan/Kaybeden/Bekleyen)</li>
                  <li>✓ Son 7 günlük kullanıcı artışı</li>
                  <li>✓ Detaylı istatistiksel veriler</li>
                </ul>
              </div>
            </CardContent>
          </Card>
        </>
      ) : null}
    </div>
  )
}
