'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { 
  Settings,
  Database,
  Key,
  Globe,
  Shield,
  Zap,
  CheckCircle,
  AlertCircle,
  Copy,
  Eye,
  EyeOff,
  RefreshCw,
  Server,
  Activity,
  Cpu,
  HardDrive,
  User,
  Lock,
  Save
} from 'lucide-react'
import { useSession } from 'next-auth/react'

export default function AdminAyarlarPage() {
  const { data: session } = useSession()
  const [showApiKey, setShowApiKey] = useState(false)
  const [showDbPassword, setShowDbPassword] = useState(false)
  const [showCurrentPassword, setShowCurrentPassword] = useState(false)
  const [showNewPassword, setShowNewPassword] = useState(false)
  const [systemInfo, setSystemInfo] = useState<any>(null)
  const [updating, setUpdating] = useState(false)
  
  // Profile update form
  const [profileData, setProfileData] = useState({
    username: '',
    email: '',
    currentPassword: '',
    newPassword: '',
    confirmPassword: ''
  })

  useEffect(() => {
    // Get system info
    setSystemInfo({
      nodeVersion: process.version || 'N/A',
      platform: typeof window !== 'undefined' ? window.navigator.platform : 'Server',
      nextVersion: '15.5.4',
      reactVersion: '19.0.0'
    })

    // Set initial profile data from session
    if (session?.user) {
      setProfileData(prev => ({
        ...prev,
        username: session.user.username || '',
        email: session.user.email || ''
      }))
    }
  }, [session])

  const copyToClipboard = (text: string, label: string) => {
    navigator.clipboard.writeText(text)
    alert(`${label} kopyalandı!`)
  }

  const handleUpdateProfile = async () => {
    if (profileData.newPassword && profileData.newPassword !== profileData.confirmPassword) {
      alert('❌ Yeni şifreler eşleşmiyor!')
      return
    }

    if (profileData.newPassword && profileData.newPassword.length < 6) {
      alert('❌ Şifre en az 6 karakter olmalı!')
      return
    }

    setUpdating(true)
    try {
      const updateData: any = {
        userId: session?.user?.id
      }

      if (profileData.username !== session?.user?.username) {
        updateData.username = profileData.username
      }

      if (profileData.email !== session?.user?.email) {
        updateData.email = profileData.email
      }

      if (profileData.newPassword && profileData.currentPassword) {
        updateData.currentPassword = profileData.currentPassword
        updateData.newPassword = profileData.newPassword
      }

      const res = await fetch('/api/admin/update-profile', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(updateData)
      })

      const data = await res.json()
      if (data.success) {
        alert('✅ Profil güncellendi! Lütfen tekrar giriş yapın.')
        setProfileData(prev => ({
          ...prev,
          currentPassword: '',
          newPassword: '',
          confirmPassword: ''
        }))
        // Refresh page to update session
        window.location.reload()
      } else {
        alert('❌ Hata: ' + (data.error || 'Profil güncellenemedi'))
      }
    } catch (error) {
      console.error('Error updating profile:', error)
      alert('❌ Profil güncellenemedi')
    } finally {
      setUpdating(false)
    }
  }

  // These would normally come from env or API
  const envConfig = {
    databaseUrl: 'mysql://iddaasohbet_iddaasohbet:****@104.247.173.212:3306/iddaasohbet_kuponlar',
    nextAuthUrl: 'http://localhost:3000',
    apiFootballKey: '807916c44ff9ddf5dcaf7cf22109b9cd'
  }

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold gradient-text mb-2">Sistem Ayarları</h1>
          <p className="text-foreground/60">Platform yapılandırması ve sistem bilgileri</p>
        </div>
        <Button 
          variant="outline" 
          className="glass border-white/10"
          onClick={() => window.location.reload()}
        >
            <RefreshCw className="h-4 w-4 mr-2" />
          Sayfayı Yenile
          </Button>
      </div>

      {/* Admin Profile Update */}
      <Card className="glass-dark border-orange-500/20">
        <CardHeader className="border-b border-white/5 bg-gradient-to-r from-orange-500/10 to-red-500/10">
          <CardTitle className="flex items-center space-x-2">
            <User className="h-5 w-5 text-orange-400" />
            <span className="gradient-text">Admin Profil Güncelleme</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          <div className="space-y-6">
            {/* Current User Info */}
            <div className="p-4 glass-dark rounded-lg border border-blue-500/20 bg-blue-500/5">
              <div className="flex items-center space-x-4">
                <div className="h-16 w-16 rounded-full bg-gradient-to-br from-orange-500 to-red-500 flex items-center justify-center text-white font-bold text-2xl">
                  {(session?.user?.username || 'A').charAt(0).toUpperCase()}
                </div>
                <div>
                  <p className="font-bold text-lg">{session?.user?.username}</p>
                  <p className="text-sm text-foreground/60">{session?.user?.email}</p>
                  <Badge className="mt-1 bg-red-500/20 border-red-500/30 text-red-400">
                    <Shield className="h-3 w-3 mr-1" />
                    Admin
                  </Badge>
                </div>
              </div>
            </div>

            {/* Update Form */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="text-sm font-semibold mb-2 block">Kullanıcı Adı</label>
                <Input
                  value={profileData.username}
                  onChange={(e) => setProfileData({...profileData, username: e.target.value})}
                  placeholder="Kullanıcı adı"
                  className="glass-dark border-white/10"
                />
              </div>
              <div>
                <label className="text-sm font-semibold mb-2 block">Email</label>
                <Input
                  type="email"
                  value={profileData.email}
                  onChange={(e) => setProfileData({...profileData, email: e.target.value})}
                  placeholder="Email adresi"
                  className="glass-dark border-white/10"
                />
              </div>
            </div>

            {/* Password Change */}
            <div className="pt-4 border-t border-white/5">
              <h3 className="font-bold mb-4 flex items-center space-x-2">
                <Lock className="h-5 w-5 text-yellow-400" />
                <span>Şifre Değiştir</span>
              </h3>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label className="text-sm font-semibold mb-2 block">Mevcut Şifre</label>
                  <div className="relative">
                    <Input
                      type={showCurrentPassword ? 'text' : 'password'}
                      value={profileData.currentPassword}
                      onChange={(e) => setProfileData({...profileData, currentPassword: e.target.value})}
                      placeholder="Mevcut şifreniz"
                      className="glass-dark border-white/10 pr-10"
                    />
                    <Button
                      size="sm"
                      variant="ghost"
                      className="absolute right-1 top-1 h-8 w-8 p-0"
                      onClick={() => setShowCurrentPassword(!showCurrentPassword)}
                    >
                      {showCurrentPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                    </Button>
                  </div>
                </div>
                <div>
                  <label className="text-sm font-semibold mb-2 block">Yeni Şifre</label>
                  <div className="relative">
                    <Input
                      type={showNewPassword ? 'text' : 'password'}
                      value={profileData.newPassword}
                      onChange={(e) => setProfileData({...profileData, newPassword: e.target.value})}
                      placeholder="Yeni şifreniz"
                      className="glass-dark border-white/10 pr-10"
                    />
                    <Button
                      size="sm"
                      variant="ghost"
                      className="absolute right-1 top-1 h-8 w-8 p-0"
                      onClick={() => setShowNewPassword(!showNewPassword)}
                    >
                      {showNewPassword ? <EyeOff className="h-4 w-4" /> : <Eye className="h-4 w-4" />}
                    </Button>
                  </div>
                </div>
            <div>
                  <label className="text-sm font-semibold mb-2 block">Şifre Tekrar</label>
                  <Input
                    type="password"
                    value={profileData.confirmPassword}
                    onChange={(e) => setProfileData({...profileData, confirmPassword: e.target.value})}
                    placeholder="Yeni şifre tekrar"
                    className="glass-dark border-white/10"
                  />
                </div>
              </div>
            </div>

            {/* Save Button */}
            <div className="pt-4">
              <Button
                onClick={handleUpdateProfile}
                disabled={updating}
                className="w-full h-12 bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-base font-bold"
              >
                <Save className="h-5 w-5 mr-2" />
                {updating ? 'Güncelleniyor...' : 'Profili Güncelle'}
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* System Status */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="glass-dark border-green-500/20">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="h-12 w-12 rounded-xl bg-green-500/10 border border-green-500/30 flex items-center justify-center">
                <Server className="h-6 w-6 text-green-400" />
              </div>
            </div>
            <p className="text-foreground/60 text-sm mb-1">Sunucu Durumu</p>
            <div className="flex items-center space-x-2">
              <div className="h-2 w-2 rounded-full bg-green-500 animate-pulse"></div>
              <p className="text-xl font-bold text-green-400">Çalışıyor</p>
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-blue-500/20">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="h-12 w-12 rounded-xl bg-blue-500/10 border border-blue-500/30 flex items-center justify-center">
                <Database className="h-6 w-6 text-blue-400" />
              </div>
            </div>
            <p className="text-foreground/60 text-sm mb-1">Veritabanı</p>
            <div className="flex items-center space-x-2">
              <CheckCircle className="h-4 w-4 text-green-400" />
              <p className="text-xl font-bold">Bağlı</p>
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-purple-500/20">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="h-12 w-12 rounded-xl bg-purple-500/10 border border-purple-500/30 flex items-center justify-center">
                <Zap className="h-6 w-6 text-purple-400" />
              </div>
            </div>
            <p className="text-foreground/60 text-sm mb-1">API Durumu</p>
            <div className="flex items-center space-x-2">
              <CheckCircle className="h-4 w-4 text-green-400" />
              <p className="text-xl font-bold">Aktif</p>
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-orange-500/20">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="h-12 w-12 rounded-xl bg-orange-500/10 border border-orange-500/30 flex items-center justify-center">
                <Activity className="h-6 w-6 text-orange-400" />
              </div>
            </div>
            <p className="text-foreground/60 text-sm mb-1">Performans</p>
            <div className="flex items-center space-x-2">
              <p className="text-xl font-bold text-orange-400">İyi</p>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Environment Configuration */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center space-x-2">
            <Settings className="h-5 w-5 text-blue-400" />
            <span>Ortam Değişkenleri</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          <div className="space-y-4">
            {/* Database URL */}
            <div className="p-4 glass-dark rounded-lg border border-white/10">
              <div className="flex items-center justify-between mb-2">
                <div className="flex items-center space-x-2">
                  <Database className="h-4 w-4 text-blue-400" />
                  <span className="font-semibold text-sm">DATABASE_URL</span>
                </div>
                <div className="flex items-center space-x-2">
                  <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">
                    <CheckCircle className="h-3 w-3 mr-1" />
                    Bağlı
                  </Badge>
                  <Button 
                    size="sm" 
                    variant="ghost"
                    onClick={() => setShowDbPassword(!showDbPassword)}
                    className="h-7 w-7 p-0"
                  >
                    {showDbPassword ? <EyeOff className="h-3 w-3" /> : <Eye className="h-3 w-3" />}
                  </Button>
                  <Button 
                    size="sm" 
                    variant="ghost"
                    onClick={() => copyToClipboard(envConfig.databaseUrl, 'Database URL')}
                    className="h-7 w-7 p-0"
                  >
                    <Copy className="h-3 w-3" />
                  </Button>
              </div>
              </div>
              <code className="text-xs text-foreground/60 font-mono break-all">
                {showDbPassword ? envConfig.databaseUrl : envConfig.databaseUrl.replace(/:[^:@]*@/, ':****@')}
              </code>
            </div>

            {/* NextAuth URL */}
            <div className="p-4 glass-dark rounded-lg border border-white/10">
              <div className="flex items-center justify-between mb-2">
                <div className="flex items-center space-x-2">
                  <Globe className="h-4 w-4 text-purple-400" />
                  <span className="font-semibold text-sm">NEXTAUTH_URL</span>
                </div>
                <Button 
                  size="sm" 
                  variant="ghost"
                  onClick={() => copyToClipboard(envConfig.nextAuthUrl, 'NextAuth URL')}
                  className="h-7 w-7 p-0"
                >
                  <Copy className="h-3 w-3" />
                </Button>
              </div>
              <code className="text-xs text-foreground/60 font-mono">{envConfig.nextAuthUrl}</code>
            </div>

            {/* Football API Key */}
            <div className="p-4 glass-dark rounded-lg border border-white/10">
              <div className="flex items-center justify-between mb-2">
                <div className="flex items-center space-x-2">
                  <Key className="h-4 w-4 text-orange-400" />
                  <span className="font-semibold text-sm">FOOTBALL API - İDDAASOHBET</span>
                </div>
                <div className="flex items-center space-x-2">
                  <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">
                    <CheckCircle className="h-3 w-3 mr-1" />
                    Aktif
                  </Badge>
                  <Button 
                    size="sm" 
                    variant="ghost"
                    onClick={() => setShowApiKey(!showApiKey)}
                    className="h-7 w-7 p-0"
                  >
                    {showApiKey ? <EyeOff className="h-3 w-3" /> : <Eye className="h-3 w-3" />}
                  </Button>
                  <Button 
                    size="sm" 
                    variant="ghost"
                    onClick={() => copyToClipboard(envConfig.apiFootballKey, 'Football API Key')}
                    className="h-7 w-7 p-0"
                  >
                    <Copy className="h-3 w-3" />
                  </Button>
                </div>
              </div>
              <code className="text-xs text-foreground/60 font-mono">
                {showApiKey ? envConfig.apiFootballKey : '••••••••••••••••••••••••••••••••'}
              </code>
              <p className="text-xs text-foreground/50 mt-2">Canlı maç skorları için API anahtarı</p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* System Information */}
      {systemInfo && (
        <Card className="glass-dark border-white/5">
          <CardHeader className="border-b border-white/5">
            <CardTitle className="flex items-center space-x-2">
              <Cpu className="h-5 w-5 text-green-400" />
              <span>Sistem Bilgileri</span>
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              <div className="p-4 glass-dark rounded-lg border border-white/10">
                <p className="text-xs text-foreground/60 mb-1">Next.js Versiyon</p>
                <p className="text-lg font-bold text-blue-400">{systemInfo.nextVersion}</p>
              </div>
              <div className="p-4 glass-dark rounded-lg border border-white/10">
                <p className="text-xs text-foreground/60 mb-1">React Versiyon</p>
                <p className="text-lg font-bold text-purple-400">{systemInfo.reactVersion}</p>
              </div>
              <div className="p-4 glass-dark rounded-lg border border-white/10">
                <p className="text-xs text-foreground/60 mb-1">Node.js Versiyon</p>
                <p className="text-lg font-bold text-green-400">v22.16.0</p>
              </div>
              <div className="p-4 glass-dark rounded-lg border border-white/10">
                <p className="text-xs text-foreground/60 mb-1">Platform</p>
                <p className="text-lg font-bold text-orange-400">{systemInfo.platform}</p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Security & Features */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Security Features */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center space-x-2">
            <Shield className="h-5 w-5 text-red-400" />
              <span>Güvenlik Özellikleri</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
            <div className="space-y-3">
              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
                <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">NextAuth Kimlik Doğrulama</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>

              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
                <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">Bcrypt Şifre Hashleme</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>

              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
              <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">Middleware Koruması</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>

              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
                <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">HTTPS/SSL</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Active Features */}
        <Card className="glass-dark border-white/5">
          <CardHeader className="border-b border-white/5">
            <CardTitle className="flex items-center space-x-2">
              <Zap className="h-5 w-5 text-yellow-400" />
              <span>Aktif Özellikler</span>
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-3">
              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
              <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">Canlı Skorlar (API-Football)</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>

              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
                <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">Canlı Sohbet Sistemi</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
            </div>

              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
              <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">Direkt Mesajlaşma</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>

              <div className="flex items-center justify-between p-3 glass rounded-lg border border-white/5">
                <div className="flex items-center space-x-3">
                  <CheckCircle className="h-5 w-5 text-green-400" />
                  <span className="text-sm">Kupon Paylaşım Sistemi</span>
                </div>
                <Badge className="bg-green-500/20 border-green-500/30 text-green-400 text-xs">Aktif</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Database Info */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center space-x-2">
            <Database className="h-5 w-5 text-blue-400" />
            <span>Veritabanı Bilgileri</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="p-4 glass-dark rounded-lg border border-white/10">
              <p className="text-xs text-foreground/60 mb-1">Tip</p>
              <p className="text-lg font-bold text-blue-400">MySQL</p>
            </div>
            <div className="p-4 glass-dark rounded-lg border border-white/10">
              <p className="text-xs text-foreground/60 mb-1">Host</p>
              <p className="text-lg font-bold text-purple-400">104.247.173.212</p>
            </div>
            <div className="p-4 glass-dark rounded-lg border border-white/10">
              <p className="text-xs text-foreground/60 mb-1">Port</p>
              <p className="text-lg font-bold text-orange-400">3306</p>
            </div>
          </div>

          <div className="mt-4 p-4 glass-dark rounded-lg border border-blue-500/20 bg-blue-500/5">
            <div className="flex items-start space-x-3">
              <AlertCircle className="h-5 w-5 text-blue-400 mt-0.5" />
              <div>
                <p className="font-semibold text-sm text-blue-400 mb-1">Önemli Not</p>
                <p className="text-xs text-foreground/70">
                  Veritabanı bilgileri güvenlik nedeniyle yalnızca görüntüleme amaçlıdır. 
                  Değişiklik yapmak için .env dosyasını düzenleyin ve sunucuyu yeniden başlatın.
                </p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Performance & Optimization */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center space-x-2">
            <Zap className="h-5 w-5 text-yellow-400" />
            <span>Performans & Optimizasyon</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="flex items-center justify-between p-4 glass rounded-lg border border-white/5">
              <div>
                <h4 className="font-semibold mb-1">Turbopack</h4>
                <p className="text-sm text-foreground/60">Next.js hızlı build sistemi</p>
              </div>
              <Badge className="bg-green-500/20 border-green-500/30 text-green-400">
                <CheckCircle className="h-3 w-3 mr-1" />
                Aktif
              </Badge>
            </div>

            <div className="flex items-center justify-between p-4 glass rounded-lg border border-white/5">
              <div>
                <h4 className="font-semibold mb-1">Prisma ORM</h4>
                <p className="text-sm text-foreground/60">Veritabanı query optimizasyonu</p>
              </div>
              <Badge className="bg-green-500/20 border-green-500/30 text-green-400">
                <CheckCircle className="h-3 w-3 mr-1" />
                Aktif
              </Badge>
            </div>

            <div className="flex items-center justify-between p-4 glass rounded-lg border border-white/5">
              <div>
                <h4 className="font-semibold mb-1">Server Components</h4>
                <p className="text-sm text-foreground/60">React Server Components</p>
              </div>
              <Badge className="bg-green-500/20 border-green-500/30 text-green-400">
                <CheckCircle className="h-3 w-3 mr-1" />
                Aktif
              </Badge>
            </div>

            <div className="flex items-center justify-between p-4 glass rounded-lg border border-white/5">
              <div>
                <h4 className="font-semibold mb-1">Image Optimization</h4>
                <p className="text-sm text-foreground/60">Next.js görsel optimizasyonu</p>
              </div>
              <Badge className="bg-green-500/20 border-green-500/30 text-green-400">
                <CheckCircle className="h-3 w-3 mr-1" />
                Aktif
              </Badge>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
}
