'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Badge } from '@/components/ui/badge'
import {
  Bot,
  Sparkles,
  Send,
  Loader2,
  CheckCircle,
  AlertCircle,
  Trophy,
  Target,
  TrendingUp
} from 'lucide-react'

export default function AITahminlerPage() {
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [creating, setCreating] = useState(false)
  const [prediction, setPrediction] = useState<any>(null)
  
  const [matchData, setMatchData] = useState({
    homeTeam: '',
    awayTeam: '',
    league: '',
    matchDate: ''
  })

  const handlePredict = async () => {
    if (!matchData.homeTeam || !matchData.awayTeam) {
      alert('Lütfen en az takım isimlerini girin')
      return
    }

    setLoading(true)
    setPrediction(null)

    try {
      const res = await fetch('/api/ai/predict', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(matchData)
      })

      if (res.ok) {
        const data = await res.json()
        setPrediction(data.prediction)
      } else {
        const error = await res.json()
        alert(error.error || 'Tahmin oluşturulamadı')
      }
    } catch (error) {
      console.error('AI tahmin hatası:', error)
      alert('Bir hata oluştu')
    } finally {
      setLoading(false)
    }
  }

  const handleCreateCoupon = async () => {
    if (!prediction) return

    setCreating(true)
    try {
      // En yüksek güvenilirliğe sahip tahminleri seç
      const selectedPredictions = prediction.predictions.filter((p: any) => 
        p.confidence === 'Yüksek'
      )

      if (selectedPredictions.length === 0) {
        alert('Yüksek güvenilirlikli tahmin bulunamadı')
        return
      }

      // Kupon oluştur
      const matches = selectedPredictions.map((p: any) => ({
        team1: matchData.homeTeam,
        team2: matchData.awayTeam,
        league: matchData.league || 'Genel',
        pick: p.prediction,
        odd: parseFloat(p.odds),
        matchDate: matchData.matchDate || new Date().toISOString()
      }))

      const totalOdds = matches.reduce((acc: number, m: any) => acc * m.odd, 1)

      const couponData = {
        title: `🤖 AI Tahmin: ${matchData.homeTeam} - ${matchData.awayTeam}`,
        description: `🤖 AI Analizi: ${prediction.analysis}\n\n⚠️ Risk Seviyesi: ${prediction.riskLevel}\n🏆 Banko Öneri: ${prediction.bankoTip}`,
        stake: 100,
        totalOdds: totalOdds.toFixed(2),
        potentialWin: (100 * totalOdds).toFixed(2),
        matches,
        aiGenerated: true
      }

      const couponRes = await fetch('/api/kuponlar', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(couponData)
      })

      if (couponRes.ok) {
        const coupon = await couponRes.json()
        alert('Kupon başarıyla oluşturuldu!')
        router.push(`/kupon/${coupon.id}`)
      } else {
        alert('Kupon oluşturulamadı')
      }
    } catch (error) {
      console.error('Kupon oluşturma hatası:', error)
      alert('Bir hata oluştu')
    } finally {
      setCreating(false)
    }
  }

  return (
    <div className="p-8 space-y-6">
      {/* Header */}
      <div className="flex items-center space-x-3 mb-8">
        <div className="relative">
          <Bot className="h-10 w-10 text-green-400 animate-pulse" />
          <Sparkles className="absolute -top-1 -right-1 h-4 w-4 text-yellow-400" />
        </div>
        <div>
          <h1 className="text-3xl font-bold gradient-text">AI Maç Tahminleri</h1>
          <p className="text-foreground/60">ChatGPT ile otomatik tahmin oluştur</p>
        </div>
      </div>

      {/* Info Card */}
      <Card className="glass-dark border-green-500/20 bg-gradient-to-r from-green-500/10 to-yellow-400/10">
        <CardContent className="p-6">
          <div className="flex items-start space-x-3">
            <AlertCircle className="h-5 w-5 text-yellow-400 mt-0.5 flex-shrink-0" />
            <div className="text-sm text-foreground/80">
              <p className="font-semibold mb-1">Nasıl Çalışır?</p>
              <p>Maç bilgilerini girin, ChatGPT analiz yaparak tahmin önerileri sunar. Ardından bu tahminleri otomatik kupon olarak oluşturabilirsiniz.</p>
            </div>
          </div>
        </CardContent>
      </Card>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Input Form */}
        <Card className="glass-dark border-white/10">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Target className="h-5 w-5 text-green-400" />
              <span>Maç Bilgileri</span>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div>
              <Label htmlFor="homeTeam">Ev Sahibi Takım *</Label>
              <Input
                id="homeTeam"
                value={matchData.homeTeam}
                onChange={(e) => setMatchData({ ...matchData, homeTeam: e.target.value })}
                placeholder="Örn: Fenerbahçe"
                className="glass border-white/10 focus:border-green-500/50"
              />
            </div>

            <div>
              <Label htmlFor="awayTeam">Deplasman Takım *</Label>
              <Input
                id="awayTeam"
                value={matchData.awayTeam}
                onChange={(e) => setMatchData({ ...matchData, awayTeam: e.target.value })}
                placeholder="Örn: Galatasaray"
                className="glass border-white/10 focus:border-green-500/50"
              />
            </div>

            <div>
              <Label htmlFor="league">Lig</Label>
              <Input
                id="league"
                value={matchData.league}
                onChange={(e) => setMatchData({ ...matchData, league: e.target.value })}
                placeholder="Örn: Süper Lig"
                className="glass border-white/10 focus:border-green-500/50"
              />
            </div>

            <div>
              <Label htmlFor="matchDate">Maç Tarihi</Label>
              <Input
                id="matchDate"
                type="datetime-local"
                value={matchData.matchDate}
                onChange={(e) => setMatchData({ ...matchData, matchDate: e.target.value })}
                className="glass border-white/10 focus:border-green-500/50"
              />
            </div>

            <Button
              onClick={handlePredict}
              disabled={loading || !matchData.homeTeam || !matchData.awayTeam}
              className="w-full bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-bold h-12"
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  AI Tahmin Oluşturuluyor...
                </>
              ) : (
                <>
                  <Bot className="mr-2 h-4 w-4" />
                  AI Tahmin Al
                </>
              )}
            </Button>
          </CardContent>
        </Card>

        {/* Results */}
        <Card className="glass-dark border-white/10">
          <CardHeader>
            <CardTitle className="flex items-center space-x-2">
              <Sparkles className="h-5 w-5 text-yellow-400" />
              <span>AI Tahmin Sonuçları</span>
            </CardTitle>
          </CardHeader>
          <CardContent>
            {!prediction ? (
              <div className="text-center py-12">
                <Bot className="h-16 w-16 mx-auto text-foreground/30 mb-4" />
                <p className="text-foreground/60">Tahmin almak için maç bilgilerini girin</p>
              </div>
            ) : (
              <div className="space-y-6">
                {/* Analysis */}
                <div className="glass p-4 rounded-xl border border-green-500/20">
                  <h4 className="font-bold text-green-400 mb-2 flex items-center space-x-2">
                    <TrendingUp className="h-4 w-4" />
                    <span>AI Analizi</span>
                  </h4>
                  <p className="text-sm text-foreground/80 leading-relaxed">{prediction.analysis}</p>
                </div>

                {/* Risk Level */}
                <div className="flex items-center justify-between glass p-4 rounded-xl border border-white/10">
                  <span className="text-sm font-medium">Risk Seviyesi:</span>
                  <Badge className={`
                    ${prediction.riskLevel === 'Düşük' ? 'bg-green-500/20 text-green-400 border-green-500/30' : ''}
                    ${prediction.riskLevel === 'Orta' ? 'bg-yellow-500/20 text-yellow-400 border-yellow-500/30' : ''}
                    ${prediction.riskLevel === 'Yüksek' ? 'bg-red-500/20 text-red-400 border-red-500/30' : ''}
                  `}>
                    {prediction.riskLevel}
                  </Badge>
                </div>

                {/* Predictions */}
                <div className="space-y-3">
                  <h4 className="font-bold text-foreground mb-3">Tahminler:</h4>
                  {prediction.predictions.map((pred: any, idx: number) => (
                    <div key={idx} className="glass p-4 rounded-xl border border-white/10 hover:border-green-500/30 transition-all">
                      <div className="flex items-center justify-between mb-2">
                        <span className="text-sm font-medium text-foreground/70">{pred.type}</span>
                        <Badge className={`
                          text-xs
                          ${pred.confidence === 'Yüksek' ? 'bg-green-500/20 text-green-400 border-green-500/30' : ''}
                          ${pred.confidence === 'Orta' ? 'bg-yellow-500/20 text-yellow-400 border-yellow-500/30' : ''}
                          ${pred.confidence === 'Düşük' ? 'bg-gray-500/20 text-gray-400 border-gray-500/30' : ''}
                        `}>
                          {pred.confidence}
                        </Badge>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="font-bold text-green-400">{pred.prediction}</span>
                        <span className="text-lg font-bold text-yellow-400">{pred.odds}</span>
                      </div>
                    </div>
                  ))}
                </div>

                {/* Banko Tip */}
                <div className="glass-dark p-4 rounded-xl border border-yellow-500/30 bg-gradient-to-r from-yellow-500/10 to-amber-500/10">
                  <div className="flex items-center space-x-2 mb-2">
                    <Trophy className="h-4 w-4 text-yellow-400" />
                    <span className="text-sm font-bold text-yellow-400">Banko Öneri</span>
                  </div>
                  <p className="text-sm text-foreground/80">{prediction.bankoTip}</p>
                </div>

                {/* Create Coupon Button */}
                <Button
                  onClick={handleCreateCoupon}
                  disabled={creating}
                  className="w-full bg-gradient-to-r from-purple-500 to-pink-500 hover:from-purple-600 hover:to-pink-600 text-white font-bold h-12"
                >
                  {creating ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Kupon Oluşturuluyor...
                    </>
                  ) : (
                    <>
                      <CheckCircle className="mr-2 h-4 w-4" />
                      Kupon Olarak Oluştur
                    </>
                  )}
                </Button>
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card className="glass-dark border-white/10 hover:border-green-500/30 transition-all">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60 mb-1">AI Model</p>
                <p className="text-2xl font-bold text-green-400">GPT-4o Mini</p>
              </div>
              <Bot className="h-10 w-10 text-green-400" />
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-white/10 hover:border-yellow-500/30 transition-all">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60 mb-1">Durum</p>
                <div className="flex items-center space-x-2">
                  <div className="h-2 w-2 bg-green-500 rounded-full animate-pulse"></div>
                  <p className="text-2xl font-bold text-white">Aktif</p>
                </div>
              </div>
              <CheckCircle className="h-10 w-10 text-green-400" />
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-white/10 hover:border-purple-500/30 transition-all">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60 mb-1">API Durumu</p>
                <p className="text-2xl font-bold text-purple-400">Hazır</p>
              </div>
              <Sparkles className="h-10 w-10 text-purple-400" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* How it Works */}
      <Card className="glass-dark border-white/10">
        <CardHeader>
          <CardTitle>Nasıl Çalışır?</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid md:grid-cols-3 gap-6">
            <div className="text-center p-6 glass rounded-xl border border-white/5">
              <div className="inline-flex items-center justify-center w-12 h-12 rounded-full bg-green-500/20 mb-4">
                <span className="text-xl font-bold text-green-400">1</span>
              </div>
              <h4 className="font-bold mb-2">Maç Bilgilerini Gir</h4>
              <p className="text-sm text-foreground/60">
                Takım isimleri, lig ve tarih bilgilerini girin
              </p>
            </div>

            <div className="text-center p-6 glass rounded-xl border border-white/5">
              <div className="inline-flex items-center justify-center w-12 h-12 rounded-full bg-yellow-500/20 mb-4">
                <span className="text-xl font-bold text-yellow-400">2</span>
              </div>
              <h4 className="font-bold mb-2">AI Analiz Yapsın</h4>
              <p className="text-sm text-foreground/60">
                ChatGPT maçı analiz eder ve tahminler üretir
              </p>
            </div>

            <div className="text-center p-6 glass rounded-xl border border-white/5">
              <div className="inline-flex items-center justify-center w-12 h-12 rounded-full bg-purple-500/20 mb-4">
                <span className="text-xl font-bold text-purple-400">3</span>
              </div>
              <h4 className="font-bold mb-2">Kupon Oluştur</h4>
              <p className="text-sm text-foreground/60">
                Tahminleri otomatik olarak kupon haline getirin
              </p>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* .env Setup Warning */}
      <Card className="glass-dark border-red-500/20 bg-gradient-to-r from-red-500/10 to-orange-500/10">
        <CardContent className="p-6">
          <div className="flex items-start space-x-3">
            <AlertCircle className="h-5 w-5 text-red-400 mt-0.5 flex-shrink-0" />
            <div className="text-sm">
              <p className="font-semibold text-red-400 mb-2">Önemli: OpenAI API Key Gerekli</p>
              <p className="text-foreground/70 mb-2">
                Bu özelliği kullanmak için .env dosyanıza OpenAI API key eklemeniz gerekiyor:
              </p>
              <code className="block bg-black/30 p-3 rounded-lg text-green-400 text-xs">
                OPENAI_API_KEY=sk-proj-your-api-key-here
              </code>
              <p className="text-foreground/60 mt-2 text-xs">
                API key almak için: <a href="https://platform.openai.com/api-keys" target="_blank" className="text-blue-400 hover:underline">platform.openai.com/api-keys</a>
              </p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  )
}

