"use client"

import { useEffect, useState } from 'react'
import Link from 'next/link'
import { Card, CardContent, CardFooter, CardHeader } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { Input } from '@/components/ui/input'
import {
  BookOpen,
  Calendar,
  Clock,
  User,
  TrendingUp,
  MessageCircle,
  Eye,
  Search,
  ArrowRight,
  Flame,
  Star,
  Target
} from 'lucide-react'

export default function BlogPage() {
  const [blogPosts, setBlogPosts] = useState<any[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    loadPosts()
  }, [])

  const loadPosts = async () => {
    try {
      const res = await fetch('/api/blog?limit=50')
      if (res.ok) {
        const data = await res.json()
        setBlogPosts(data.posts || [])
      }
    } catch (error) {
      console.error('Blog yazıları yüklenirken hata:', error)
    } finally {
      setLoading(false)
    }
  }

  const featuredPosts = blogPosts.slice(0, 3)
  const regularPosts = blogPosts.slice(3)

  return (
    <div className="min-h-screen py-12 relative">
      <div className="absolute inset-0 grid-pattern"></div>
      <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-purple-500/5 rounded-full blur-3xl animate-pulse-slow"></div>
      <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-blue-500/5 rounded-full blur-3xl animate-pulse-slow" style={{ animationDelay: '1s' }}></div>

      <div className="container mx-auto px-4 relative z-10">
        <div className="text-center mb-12 animate-fadeInUp">
          <div className="flex items-center justify-center space-x-3 mb-4">
            <BookOpen className="h-10 w-10 text-purple-400 animate-pulse" />
            <h1 className="text-4xl md:text-5xl font-bold gradient-text">Blog & İçerikler</h1>
          </div>
          <p className="text-foreground/70 text-lg max-w-2xl mx-auto">
            İddaa stratejileri, analizler ve ipuçları ile başarınızı artırın
          </p>
        </div>

        {!loading && blogPosts.length > 0 && (
          <>
            <div className="max-w-2xl mx-auto mb-12 animate-fadeInUp" style={{ animationDelay: '0.1s' }}>
              <div className="relative">
                <Search className="absolute left-4 top-1/2 -translate-y-1/2 h-5 w-5 text-foreground/50" />
                <Input type="text" placeholder="Makale ara..." className="pl-12 glass border-white/10 focus:border-green-500/50 bg-black/20 h-14 text-base" />
              </div>
            </div>
          </>
        )}

        {loading ? (
          <div className="text-center py-12">
            <div className="inline-block h-12 w-12 animate-spin rounded-full border-4 border-solid border-green-400 border-r-transparent"></div>
            <p className="mt-4 text-foreground/60">Yükleniyor...</p>
          </div>
        ) : blogPosts.length === 0 ? (
          <Card className="glass-dark border-white/10 p-12 text-center">
            <div className="flex flex-col items-center space-y-4">
              <BookOpen className="h-20 w-20 text-foreground/30" />
              <div>
                <h3 className="text-2xl font-semibold mb-2">Henüz Blog Yazısı Yok</h3>
                <p className="text-foreground/60">Yakında harika içeriklerle buradayız!</p>
              </div>
            </div>
          </Card>
        ) : (
          <>
            {featuredPosts.length > 0 && (
              <div className="mb-16">
                <div className="flex items-center space-x-3 mb-6 animate-fadeInUp" style={{ animationDelay: '0.3s' }}>
                  <Flame className="h-6 w-6 text-orange-500 animate-pulse" />
                  <h2 className="text-2xl font-bold">Öne Çıkan Yazılar</h2>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                  {featuredPosts.map((post, index) => (
                    <Card key={post.id} className="glass-dark border-white/5 card-premium group animate-fadeInUp" style={{ animationDelay: `${0.4 + index * 0.1}s` }}>
                      <CardHeader className="pb-3">
                        <div className="flex items-center justify-between mb-4">
                          <Badge className="bg-gradient-to-r from-orange-500 to-red-500 text-white border-0">
                            <Flame className="h-3 w-3 mr-1" />
                            Öne Çıkan
                          </Badge>
                        </div>
                        <h3 className="text-xl font-bold leading-tight group-hover:text-green-400 transition-colors">{post.title}</h3>
                      </CardHeader>
                      <CardContent className="pb-3">
                        <p className="text-sm text-foreground/70 line-clamp-3 mb-4">{post.excerpt}</p>
                        <Badge variant="outline" className="glass border-green-500/30 text-green-400 text-xs">{post.category}</Badge>
                      </CardContent>
                      <CardFooter className="pt-3 border-t border-white/5">
                        <div className="flex items-center justify-between w-full text-xs text-foreground/60">
                          <div className="flex items-center space-x-3">
                            <div className="flex items-center space-x-1">
                              <Eye className="h-3 w-3" />
                              <span>{post.views}</span>
                            </div>
                          </div>
                          <div className="flex items-center space-x-1">
                            <Clock className="h-3 w-3" />
                            <span>5 dakika</span>
                          </div>
                        </div>
                      </CardFooter>
                    </Card>
                  ))}
                </div>
              </div>
            )}
          </>
        )}

        {!loading && regularPosts.length > 0 && (
          <div>
            <div className="flex items-center space-x-3 mb-6 animate-fadeInUp">
              <BookOpen className="h-6 w-6 text-blue-400" />
              <h2 className="text-2xl font-bold">Son Yazılar</h2>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {regularPosts.map((post, index) => (
                <Link key={post.id} href={`/blog/${post.id}`}>
                  <Card className="glass-dark border-white/5 card-premium group cursor-pointer animate-fadeInUp" style={{ animationDelay: `${0.1 + index * 0.05}s` }}>
                    <CardContent className="p-6">
                      <div className="flex items-start space-x-4">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center space-x-2 mb-2">
                            <Badge variant="outline" className="glass border-green-500/30 text-green-400 text-xs">{post.category}</Badge>
                            <span className="text-xs text-foreground/50">•</span>
                            <div className="flex items-center space-x-1 text-xs text-foreground/50">
                              <Calendar className="h-3 w-3" />
                              <span>{new Date(post.createdAt).toLocaleDateString('tr-TR')}</span>
                            </div>
                          </div>

                          <h3 className="text-lg font-bold mb-2 group-hover:text-green-400 transition-colors leading-tight">{post.title}</h3>
                          <p className="text-sm text-foreground/70 line-clamp-2 mb-3">{post.excerpt}</p>

                          <div className="flex items-center justify-between text-xs text-foreground/60">
                            <div className="flex items-center space-x-1">
                              <User className="h-3 w-3" />
                              <span>{post.author}</span>
                            </div>
                            <div className="flex items-center space-x-3">
                              <div className="flex items-center space-x-1">
                                <Eye className="h-3 w-3" />
                                <span>{post.views}</span>
                              </div>
                              <div className="flex items-center space-x-1">
                                <Clock className="h-3 w-3" />
                                <span>5 dakika</span>
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                </Link>
              ))}
            </div>
          </div>
        )}

        {!loading && blogPosts.length > 6 && (
          <div className="text-center mt-12 animate-fadeInUp">
            <Button size="lg" className="bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-bold btn-premium px-12">
              Daha Fazla Yükle
              <ArrowRight className="ml-2 h-5 w-5" />
            </Button>
          </div>
        )}

        {!loading && blogPosts.length > 0 && (
          <Card className="glass-dark border-white/5 mt-16 animate-fadeInUp">
            <CardContent className="p-12 text-center">
              <div className="max-w-2xl mx-auto">
                <BookOpen className="h-12 w-12 text-green-400 mx-auto mb-4" />
                <h3 className="text-2xl font-bold mb-3 gradient-text">Yeni İçeriklerden Haberdar Olun</h3>
                <p className="text-foreground/70 mb-6">E-posta listemize katılın ve yeni blog yazıları hakkında bildirim alın</p>
                <div className="flex flex-col sm:flex-row gap-3 max-w-md mx-auto">
                  <Input type="email" placeholder="E-posta adresiniz" className="glass border-white/10 focus:border-green-500/50 bg-black/20 h-12 flex-1" />
                  <Button className="bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-bold btn-premium h-12 px-8">Abone Ol</Button>
                </div>
              </div>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  )
}
