'use client'

import { useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import { Badge } from '@/components/ui/badge'
import {
  Bot,
  Sparkles,
  Loader2,
  Trophy,
  Target,
  TrendingUp,
  AlertCircle,
  Brain,
  Zap
} from 'lucide-react'

export default function AITahminPage() {
  const [loading, setLoading] = useState(false)
  const [progress, setProgress] = useState(0)
  const [prediction, setPrediction] = useState<any>(null)
  const [chatHistory, setChatHistory] = useState<any[]>([])
  
  const [matchData, setMatchData] = useState({
    homeTeam: '',
    awayTeam: '',
    league: '',
    matchDate: ''
  })

  const handlePredict = async () => {
    if (!matchData.homeTeam || !matchData.awayTeam) {
      alert('Lütfen en az takım isimlerini girin')
      return
    }

    setLoading(true)
    setPrediction(null)
    setProgress(0)

    // Progress bar animasyonu
    const progressInterval = setInterval(() => {
      setProgress(prev => {
        if (prev >= 95) {
          clearInterval(progressInterval)
          return 95
        }
        return prev + 5
      })
    }, 100)

    // Chat history'e soruyu ekle
    const question = {
      type: 'user',
      content: `${matchData.homeTeam} - ${matchData.awayTeam} maçı için tahmin?`,
      timestamp: new Date()
    }
    setChatHistory([...chatHistory, question])

    try {
      const res = await fetch('/api/ai/predict', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(matchData)
      })

      if (res.ok) {
        const data = await res.json()
        clearInterval(progressInterval)
        setProgress(100)
        
        // Kısa bir gecikme ile sonucu göster
        setTimeout(() => {
          setPrediction(data.prediction)
          
          // Chat history'e cevabı ekle
          setChatHistory(prev => [...prev, {
            type: 'ai',
            content: data.prediction,
            timestamp: new Date()
          }])
        }, 500)
      } else {
        const error = await res.json()
        clearInterval(progressInterval)
        alert(error.error || 'Tahmin oluşturulamadı')
        // Hatayı history'den kaldır
        setChatHistory(prev => prev.slice(0, -1))
      }
    } catch (error) {
      console.error('AI tahmin hatası:', error)
      clearInterval(progressInterval)
      alert('Bir hata oluştu')
      setChatHistory(prev => prev.slice(0, -1))
    } finally {
      setLoading(false)
      setProgress(0)
    }
  }

  const resetForm = () => {
    setMatchData({ homeTeam: '', awayTeam: '', league: '', matchDate: '' })
    setPrediction(null)
  }

  return (
    <div className="min-h-screen py-12 relative">
      {/* Background */}
      <div className="absolute inset-0 grid-pattern"></div>
      <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-purple-500/10 rounded-full blur-3xl animate-pulse-slow"></div>
      <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-pink-500/10 rounded-full blur-3xl animate-pulse-slow" style={{ animationDelay: '1s' }}></div>

      <div className="container mx-auto px-4 relative z-10 max-w-6xl">
        {/* Header */}
        <div className="text-center mb-12 animate-fadeInUp">
          <h1 className="text-5xl md:text-6xl font-black mb-4">
            <span className="text-green-400">İddaa</span>
            <span className="text-yellow-400">Sohbet</span>
            {' '}
            <span className="gradient-text">AI Analiz</span>
          </h1>
          <p className="text-xl text-foreground/70 max-w-2xl mx-auto">
            Profesyonel yapay zeka destekli maç analizleri. Maç bilgilerini girin, <span className="text-purple-400 font-bold">İddaaSohbet AI</span> sistemi analiz etsin!
          </p>
        </div>

        {/* Input Form - Full Width, Single Column */}
        {!prediction && (
          <Card className="glass-dark border-white/10 animate-fadeInUp">
            <CardHeader className="text-center pb-8">
              <CardTitle className="flex items-center justify-center space-x-3 text-2xl">
                <Target className="h-7 w-7 text-green-400" />
                <span>Maç Bilgilerini Girin</span>
              </CardTitle>
              <p className="text-foreground/60 mt-3">İddaaSohbet AI sistemi detaylı analiz yapacak</p>
            </CardHeader>
            <CardContent>
              {/* Horizontal Form */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                <div>
                  <Label htmlFor="homeTeam" className="text-base font-semibold mb-2 block">Ev Sahibi Takım *</Label>
                  <Input
                    id="homeTeam"
                    value={matchData.homeTeam}
                    onChange={(e) => setMatchData({ ...matchData, homeTeam: e.target.value })}
                    placeholder="Örn: Fenerbahçe, Manchester City"
                    className="glass border-white/10 focus:border-purple-500/50 h-14 text-lg"
                    disabled={loading}
                  />
                </div>

                <div>
                  <Label htmlFor="awayTeam" className="text-base font-semibold mb-2 block">Deplasman Takım *</Label>
                  <Input
                    id="awayTeam"
                    value={matchData.awayTeam}
                    onChange={(e) => setMatchData({ ...matchData, awayTeam: e.target.value })}
                    placeholder="Örn: Galatasaray, Real Madrid"
                    className="glass border-white/10 focus:border-purple-500/50 h-14 text-lg"
                    disabled={loading}
                  />
                </div>

                <div>
                  <Label htmlFor="league" className="text-base font-semibold mb-2 block">Lig (Opsiyonel)</Label>
                  <Input
                    id="league"
                    value={matchData.league}
                    onChange={(e) => setMatchData({ ...matchData, league: e.target.value })}
                    placeholder="Örn: Süper Lig, Premier League"
                    className="glass border-white/10 focus:border-purple-500/50 h-14 text-lg"
                    disabled={loading}
                  />
                </div>

                <div>
                  <Label htmlFor="matchDate" className="text-base font-semibold mb-2 block">Maç Tarihi (Opsiyonel)</Label>
                  <Input
                    id="matchDate"
                    type="datetime-local"
                    value={matchData.matchDate}
                    onChange={(e) => setMatchData({ ...matchData, matchDate: e.target.value })}
                    className="glass border-white/10 focus:border-purple-500/50 h-14 text-lg"
                    disabled={loading}
                  />
                </div>
              </div>

              {/* Progress Bar - Görünür sadece loading true iken */}
              {loading && (
                <div className="mb-6 space-y-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-2">
                      <Bot className="h-5 w-5 text-purple-400 animate-pulse" />
                      <span className="text-sm font-semibold text-purple-400">İddaaSohbet AI Analiz Yapıyor...</span>
                    </div>
                    <span className="text-lg font-black text-purple-400">{progress}%</span>
                  </div>
                  <div className="relative h-3 bg-black/30 rounded-full overflow-hidden border border-purple-500/30">
                    <div 
                      className="absolute inset-y-0 left-0 bg-gradient-to-r from-purple-500 via-pink-500 to-purple-500 rounded-full transition-all duration-300 ease-out"
                      style={{ width: `${progress}%` }}
                    >
                      <div className="absolute inset-0 bg-gradient-to-r from-white/0 via-white/30 to-white/0 animate-pulse"></div>
                    </div>
                  </div>
                  <p className="text-xs text-center text-foreground/60">
                    Maç analizi yapılıyor, lütfen bekleyin...
                  </p>
                </div>
              )}

              {/* Action Button */}
              <Button
                onClick={handlePredict}
                disabled={loading || !matchData.homeTeam || !matchData.awayTeam}
                className="w-full bg-gradient-to-r from-purple-500 via-pink-500 to-purple-500 hover:from-purple-600 hover:via-pink-600 hover:to-purple-600 text-white font-black h-16 text-xl shadow-2xl shadow-purple-500/50 hover:shadow-purple-500/80 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
              >
                <Bot className="mr-3 h-6 w-6" />
                {loading ? 'Analiz Ediliyor...' : 'İddaaSohbet AI ile Analiz Et'}
                <Sparkles className="ml-3 h-5 w-5" />
              </Button>
            </CardContent>
          </Card>
        )}

          {/* Results - Center when shown */}
          {prediction && (
            <div className="space-y-8">
              {/* Header with Back Button */}
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <h2 className="text-3xl font-black gradient-text mb-2">
                    {matchData.homeTeam} vs {matchData.awayTeam}
                  </h2>
                  <p className="text-foreground/60">İddaaSohbet AI Analiz Sonucu</p>
                </div>
                <Button
                  onClick={resetForm}
                  variant="outline"
                  className="border-white/20 hover:border-green-500/50 hover:bg-green-500/10"
                >
                  Yeni Analiz Yap
                </Button>
              </div>

              <div className="space-y-6">
                {/* Analysis */}
                  <div className="relative glass-dark p-8 rounded-2xl border-2 border-purple-500/40 bg-gradient-to-br from-purple-500/10 to-pink-500/10 overflow-hidden">
                    <div className="absolute inset-0 bg-gradient-to-br from-purple-500/5 to-pink-500/5"></div>
                    <div className="relative">
                      <div className="flex items-start space-x-4 mb-4">
                        <div className="flex-shrink-0 w-12 h-12 rounded-xl bg-gradient-to-br from-purple-500 to-pink-500 flex items-center justify-center shadow-lg shadow-purple-500/50">
                          <Brain className="h-6 w-6 text-white" />
                        </div>
                        <div className="flex-1">
                          <h4 className="font-black text-xl text-transparent bg-gradient-to-r from-purple-300 to-pink-300 bg-clip-text mb-1">
                            İddaaSohbet AI Maç Analizi
                          </h4>
                          <p className="text-sm text-foreground/70">Profesyonel Yapay Zeka Değerlendirmesi</p>
                        </div>
                      </div>
                      <div className="bg-black/30 p-4 rounded-xl border border-white/10">
                        <p className="text-foreground/90 leading-relaxed text-base">{prediction.analysis}</p>
                      </div>
                    </div>
                  </div>

                  {/* PRO Maç Tahmini - Anasayfa Stili */}
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    {prediction.predictions.map((pred: any, idx: number) => (
                      <Card key={idx} className="glass-dark border-white/10 hover:border-green-500/30 transition-all group">
                        <CardContent className="p-8 text-center">
                          <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-green-500/10 mb-4 group-hover:bg-green-500/20 transition-all">
                            <Target className="h-8 w-8 text-green-400" />
                          </div>
                          <h3 className="text-sm text-foreground/60 mb-2">{pred.type}</h3>
                          <div className="text-4xl font-black mb-3 gradient-text">
                            {pred.prediction}
                          </div>
                          <Badge className={`
                            ${pred.confidence === 'Yüksek' ? 'bg-green-500/20 text-green-400 border-green-500/30' : ''}
                            ${pred.confidence === 'Orta' ? 'bg-yellow-500/20 text-yellow-400 border-yellow-500/30' : ''}
                            ${pred.confidence === 'Düşük' ? 'bg-gray-500/20 text-gray-400 border-gray-500/30' : ''}
                            font-bold
                          `}>
                            {pred.confidence} Güven
                          </Badge>
                        </CardContent>
                      </Card>
                    ))}
                  </div>

                  {/* Banko Önerisi - Anasayfa Stili */}
                  <Card className="glass-dark border-white/10 relative overflow-hidden">
                    <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-yellow-400/10"></div>
                    <CardContent className="p-12 text-center relative z-10">
                      <Trophy className="h-16 w-16 text-yellow-400 mx-auto mb-6 animate-pulse" />
                      <h2 className="text-4xl font-bold mb-4 gradient-text">İddaaSohbet AI - Banko Önerisi</h2>
                      <p className="text-xl text-foreground/80 mb-4">
                        {prediction.bankoTip}
                      </p>
                      <Badge className={`
                        ${prediction.riskLevel === 'Düşük' ? 'bg-green-500/20 text-green-400 border-green-500/50' : ''}
                        ${prediction.riskLevel === 'Orta' ? 'bg-yellow-500/20 text-yellow-400 border-yellow-500/50' : ''}
                        ${prediction.riskLevel === 'Yüksek' ? 'bg-red-500/20 text-red-400 border-red-500/50' : ''}
                        font-bold text-lg px-6 py-2
                      `}>
                        Risk Seviyesi: {prediction.riskLevel}
                      </Badge>
                    </CardContent>
                  </Card>

                  {/* Disclaimer */}
                  <div className="glass-dark p-5 rounded-xl border border-red-500/30 bg-red-500/5">
                    <div className="flex items-start space-x-3">
                      <AlertCircle className="h-5 w-5 text-red-400 mt-0.5 flex-shrink-0" />
                      <div>
                        <p className="text-sm font-bold text-red-400 mb-1">⚠️ Önemli Uyarı</p>
                        <p className="text-sm text-foreground/80 leading-relaxed">
                          Bu tahminler <span className="font-bold text-purple-400">İddaaSohbet AI</span> yapay zeka sistemi tarafından oluşturulmuştur. 
                          Kesin sonuç garantisi yoktur. Bahis kararlarınız tamamen size aittir. <span className="font-bold">18+ Sorumlu oyun.</span>
                        </p>
                      </div>
                    </div>
                  </div>
              </div>
            </div>
          )}

        {/* Stats Section */}
        <section className="mt-16" aria-labelledby="features">
          <h2 id="features" className="sr-only">Platform Özellikleri</h2>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Card className="glass-dark border-white/10 hover:border-green-500/30 transition-all group">
              <CardContent className="p-8 text-center">
                <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-green-500/10 mb-4 group-hover:bg-green-500/20 transition-all">
                  <Zap className="h-8 w-8 text-green-400" />
                </div>
                <h3 className="text-3xl font-bold mb-2 gradient-text">Anında Analiz</h3>
                <p className="text-foreground/60">
                  İddaaSohbet AI ile saniyeler içinde detaylı maç analizi ve tahmin sonuçları alın
                </p>
              </CardContent>
            </Card>

            <Card className="glass-dark border-white/10 hover:border-yellow-400/30 transition-all group">
              <CardContent className="p-8 text-center">
                <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-yellow-400/10 mb-4 group-hover:bg-yellow-400/20 transition-all">
                  <Brain className="h-8 w-8 text-yellow-400" />
                </div>
                <h3 className="text-3xl font-bold mb-2 gradient-text">Yapay Zeka Gücü</h3>
                <p className="text-foreground/60">
                  Gelişmiş AI algoritmaları ile tüm maçlar için profesyonel tahmin üretimi
                </p>
              </CardContent>
            </Card>

            <Card className="glass-dark border-white/10 hover:border-green-500/30 transition-all group">
              <CardContent className="p-8 text-center">
                <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-green-500/10 mb-4 group-hover:bg-green-500/20 transition-all">
                  <Trophy className="h-8 w-8 text-green-400" />
                </div>
                <h3 className="text-3xl font-bold mb-2 gradient-text">Banko Öneriler</h3>
                <p className="text-foreground/60">
                  Her analiz için AI'ın seçtiği en güvenilir ve yüksek ihtimalli tahminler
                </p>
              </CardContent>
            </Card>
          </div>
        </section>
      </div>
    </div>
  )
}

