const { EmbedBuilder } = require('discord.js');
const fs = require('fs');

/* =========================
   🧠 Yardımcılar
========================= */

// Leet + parçalama normalize (a-l-l-a-h, @llah, a l l a h => hepsi aynı)
function normalizeNick(input) {
  return String(input || '')
    .toLowerCase()
    .replace(/[@4]/g, 'a')
    .replace(/[3]/g, 'e')
    .replace(/[1!|]/g, 'i')
    .replace(/[0]/g, 'o')
    .replace(/[5$]/g, 's')
    .replace(/[7]/g, 't')
    .replace(/[^a-z0-9]/g, ''); // tüm boşluk/işaretleri sil
}

// nick DB (kalıcı)
function loadNickler() {
  try {
    if (fs.existsSync('./nickler.json')) {
      return JSON.parse(fs.readFileSync('./nickler.json', 'utf-8'));
    }
  } catch {}
  return {};
}

function saveNickler(data) {
  try {
    fs.writeFileSync('./nickler.json', JSON.stringify(data, null, 2));
  } catch {}
}

// aynı kullanıcının eski nick kayıtlarını temizle
function removeUserNickFromGuild(nickDb, guildId, userId, keepKey = null) {
  if (!nickDb[guildId]) nickDb[guildId] = {};
  for (const k of Object.keys(nickDb[guildId])) {
    if (nickDb[guildId][k] === userId && k !== keepKey) delete nickDb[guildId][k];
  }
}

/* =========================
   🚫 KARA LİSTE
========================= */

const badWords = [
  // 🔧 Yetki / sistem troll
  'admin','mod','moderator','yetkili','owner','kurucu','founder',
  'support','destek','staff','developer','dev','root','system',
  'discord','dc','bot','security','guard','ceo',

  // 🤡 Troll / provoke (muhammed/muhammet YOK)
  'allah','tanri','tanrı','peygamber',
  'islam','musluman','müslüman','kuran','kur’an',
  'hz','hazreti','resul','rasul',
  'atatürk','atatürkçü','kemal','mustafakemal',
  'hitler','stalin','lenin','putin',

  // 🌐 Platform / forum troll
  'doeda','doeda2','doeda3','doedanet',
  'onlyfans','only','fans','tiktok','instagram','youtube',

  // 🚫 Küfür / argo (TR)
  'amk','aq','mk','oc','oç','orospu','pic','piç',
  'sik','siktir','sikerim','sikim','yarrak','yarak',
  'anan','baban','ibne','gavat','pezevenk',
  'amina','amına','yavsak','salak','gerizekali','gerizekalı',
  'mal','aptal','embesil',

  // 🚫 Küfür / argo (EN)
  'fuck','fucker','shit','bitch','slut','whore',
  'dick','cock','pussy','asshole','bastard',
  'nigger','nigga','retard',

  // 🔞 NSFW
  'sex','seks','porno','porn','nsfw','xxx',
  'anal','oral','blowjob','handjob',
  'escort','fetish','hentai',

  // 🧨 Spam / scam
  'free','bedava','gift','nitro','steam','csgo','valorant',
  'hack','cheat','hile','exploit','crack','leak',

  // 🎮 Senin istediklerin
  'leet','hubbe','swift','tabbo','runo',
  'heel','hsmile',
  'hablon','hablos','habron','habbiz'
];

/* =========================
   🔒 Nick kontrolü (harf+sayı, tek kelime)
========================= */
function nicknameGecerliMi(nick) {
  if (!nick) return { ok: false, reason: 'Nick boş olamaz.' };

  const raw = nick.trim();

  if (/\s/.test(raw)) return { ok: false, reason: 'Nick boşluk içeremez.' };
  if (raw.length < 1) return { ok: false, reason: 'En az 1 karakter olmalı.' };
  if (raw.length > 24) return { ok: false, reason: 'En fazla 24 karakter olmalı.' };

  const allowedRegex = /^[a-zA-Z0-9]+$/;
  if (!allowedRegex.test(raw)) return { ok: false, reason: 'Sadece harf ve sayı kullanabilirsin.' };

  if (/(.)\1{3,}/i.test(raw)) return { ok: false, reason: 'Aşırı tekrar içeremez.' };

  // normalize kontrol (parçalayarak yazmayı engeller)
  const normalized = normalizeNick(raw);
  if (badWords.some(w => normalized.includes(w))) {
    return { ok: false, reason: 'Uygunsuz / yasaklı kelime içeriyor.' };
  }

  return { ok: true, nick: raw };
}

/* =========================
   🧠 Spam sayaçları
========================= */

const invalidAttempts = new Map(); // guildId:userId => {count, last}
const MAX_INVALID = 3;
const WINDOW_MS = 60_000;      // 1 dk
const TIMEOUT_MS = 5 * 60_000; // 5 dk

function keyOf(guildId, userId) {
  return `${guildId}:${userId}`;
}

function bumpInvalid(guildId, userId) {
  const k = keyOf(guildId, userId);
  const now = Date.now();
  const prev = invalidAttempts.get(k);

  if (!prev || (now - prev.last) > WINDOW_MS) {
    invalidAttempts.set(k, { count: 1, last: now });
    return 1;
  }

  const next = { count: prev.count + 1, last: now };
  invalidAttempts.set(k, next);
  return next.count;
}

function resetInvalid(guildId, userId) {
  invalidAttempts.delete(keyOf(guildId, userId));
}

/* =========================
   🎯 EVENT
========================= */

module.exports = {
  name: 'messageCreate',
  async execute(message, client) {
    if (message.author.bot) return;
    if (!message.guild) return;

    try {
      const ayarlar = client.loadAyarlar();
      const guildSettings = ayarlar[message.guild.id];
      if (!guildSettings) return;

      // sadece kayıt kanalı
      if (message.channel.id !== guildSettings.kayitKanal) return;

      const kayitsizRol = message.guild.roles.cache.get(guildSettings.kayitsizRol);
      if (!message.member.roles.cache.has(kayitsizRol?.id)) return;

      const rawNick = message.content;
      const normalized = normalizeNick(rawNick);

      // ⛔ Kara listeye takıldıysa: direkt 5 dk timeout
      if (badWords.some(w => normalized.includes(w))) {
        await message.delete().catch(() => {});
        await message.member.timeout(TIMEOUT_MS, 'Yasaklı / troll nick').catch(() => {});

        const embed = new EmbedBuilder()
          .setColor('#ff4c4c')
          .setTitle('⛔ Timeout')
          .setDescription('Kanka bu nick yasaklı/troll. **5 dakika timeout** geldi.')
          .setTimestamp();

        const tmsg = await message.channel
          .send({ content: `${message.author}`, embeds: [embed] })
          .catch(() => null);

        if (tmsg) setTimeout(() => tmsg.delete().catch(() => {}), 10_000);
        return;
      }

      // normal geçersiz kontrol
      const kontrol = nicknameGecerliMi(rawNick);

      if (!kontrol.ok) {
        const count = bumpInvalid(message.guild.id, message.author.id);
        await message.delete().catch(() => {});

        if (count >= MAX_INVALID) {
          resetInvalid(message.guild.id, message.author.id);
          await message.member.timeout(TIMEOUT_MS, 'Geçersiz nick spam (3 kez)').catch(() => {});

          const embed = new EmbedBuilder()
            .setColor('#ff4c4c')
            .setTitle('⏳ Timeout')
            .setDescription('Kanka spam yapma 😄 **5 dakika timeout** geldi.')
            .setTimestamp();

          const tmsg = await message.channel
            .send({ content: `${message.author}`, embeds: [embed] })
            .catch(() => null);

          if (tmsg) setTimeout(() => tmsg.delete().catch(() => {}), 10_000);
          return;
        }

        const uyari = new EmbedBuilder()
          .setColor('#ff4c4c')
          .setTitle('⚠️ Geçersiz Nick')
          .setDescription(
            `Sebep: **${kontrol.reason}**\n` +
            `Deneme: **${count}/${MAX_INVALID}** (1 dk içinde)\n\n` +
            `Örnek: \`rilaks\`, \`fu\`, \`lillian\`, \`a\`, \`x7\``
          )
          .setTimestamp();

        const warnMsg = await message.channel
          .send({ content: `${message.author}`, embeds: [uyari] })
          .catch(() => null);

        if (warnMsg) setTimeout(() => warnMsg.delete().catch(() => {}), 10_000);
        return;
      }

      // ✅ BURADA nick geçerli -> artık "tek kişi alsın" kontrolü
      const nickDb = loadNickler();
      const gid = message.guild.id;
      const uid = message.author.id;
      if (!nickDb[gid]) nickDb[gid] = {};

      const reservedBy = nickDb[gid][normalized];

      // başkası aldıysa engelle
      if (reservedBy && reservedBy !== uid) {
        await message.delete().catch(() => {});

        const embed = new EmbedBuilder()
          .setColor('#ff4c4c')
          .setTitle('⚠️ Nick Kullanımda')
          .setDescription('Kanka bu nick alınmış. Başka bir nick dene.')
          .setTimestamp();

        const m = await message.channel
          .send({ content: `${message.author}`, embeds: [embed] })
          .catch(() => null);

        if (m) setTimeout(() => m.delete().catch(() => {}), 10_000);
        return;
      }

      // ✅ başarılı kayıt
      resetInvalid(message.guild.id, message.author.id);

      const yeniNick = kontrol.nick;
      const uyeRol = message.guild.roles.cache.get(guildSettings.uyeRol);

      await message.member.setNickname(yeniNick).catch(() => {});
      if (kayitsizRol) await message.member.roles.remove(kayitsizRol).catch(() => {});
      if (uyeRol) await message.member.roles.add(uyeRol).catch(() => {});
      await message.delete().catch(() => {});

      // Hoşgeldin embedini sil
      if (client.kayitWelcomeMsgs) {
        const k = `${message.guild.id}:${message.author.id}`;
        const welcomeMsgId = client.kayitWelcomeMsgs.get(k);
        if (welcomeMsgId) {
          const msg = await message.channel.messages.fetch(welcomeMsgId).catch(() => null);
          if (msg) await msg.delete().catch(() => {});
          client.kayitWelcomeMsgs.delete(k);
        }
      }

      // ✅ Nick DB: kullanıcı başka nick aldıysa eskilerini temizle, bu nicki ona kilitle
      removeUserNickFromGuild(nickDb, gid, uid, normalized);
      nickDb[gid][normalized] = uid;
      saveNickler(nickDb);

      // tam sessiz
      return;

    } catch (error) {
      console.error('Kayıt hatası:', error);
    }
  }
};
