require('dotenv').config();
const {
  Client,
  GatewayIntentBits,
  Collection,
  REST,
  Routes,
  ActivityType
} = require('discord.js');
const {
  joinVoiceChannel,
  getVoiceConnection,
  VoiceConnectionStatus,
  entersState
} = require('@discordjs/voice');
const fs = require('fs');

const client = new Client({
  intents: [
    GatewayIntentBits.Guilds,
    GatewayIntentBits.GuildMembers,
    GatewayIntentBits.GuildMessages,
    GatewayIntentBits.MessageContent,
    GatewayIntentBits.GuildVoiceStates,
  ],
});

client.commands = new Collection();

/* ───────────────────────────────
   Ayarları yükleme / kaydetme
─────────────────────────────── */
function loadAyarlar() {
  try {
    if (fs.existsSync('./ayarlar.json')) {
      return JSON.parse(fs.readFileSync('./ayarlar.json', 'utf-8'));
    }
  } catch (error) {
    console.error('Ayarlar yüklenirken hata:', error);
  }
  return {};
}

function saveAyarlar(data) {
  try {
    fs.writeFileSync('./ayarlar.json', JSON.stringify(data, null, 2));
  } catch (error) {
    console.error('Ayarlar kaydedilirken hata:', error);
  }
}

client.loadAyarlar = loadAyarlar;
client.saveAyarlar = saveAyarlar;

/* ───────────────────────────────
   Komutları yükle
─────────────────────────────── */
const commandFiles = fs.readdirSync('./commands').filter(f => f.endsWith('.js'));
const commands = [];

for (const file of commandFiles) {
  const command = require(`./commands/${file}`);
  client.commands.set(command.data.name, command);
  commands.push(command.data.toJSON());
}

/* ───────────────────────────────
   Eventleri yükle
─────────────────────────────── */
const eventFiles = fs.readdirSync('./events').filter(f => f.endsWith('.js'));
for (const file of eventFiles) {
  const event = require(`./events/${file}`);
  if (event.once) {
    client.once(event.name, (...args) => event.execute(...args, client));
  } else {
    client.on(event.name, (...args) => event.execute(...args, client));
  }
}

/* ───────────────────────────────
   Durum mesajları (modern)
─────────────────────────────── */
const statusMessages = [
  { name: 'rilaks ❤️ habnet', type: ActivityType.Watching },
  { name: 'H A B N E T', type: ActivityType.Watching },
  { name: 'discord.gg/habnetresmi', type: ActivityType.Playing },
  { name: 'https://habnet.com.tr/', type: ActivityType.Playing },
  { name: 'rilaks ❤️ habnet', type: ActivityType.Playing },
  { name: 'rilaks ❤️ habnet', type: ActivityType.Listening },
  { name: 'rilaks ❤️ habnet', type: ActivityType.Listening }
];

let currentStatusIndex = 0;
function updateStatus() {
  if (!client.user) return;
  client.user.setPresence({
    activities: [statusMessages[currentStatusIndex]],
    status: 'online',
  });
  currentStatusIndex = (currentStatusIndex + 1) % statusMessages.length;
}

/* ───────────────────────────────
   Ses kanalına bağlanma
─────────────────────────────── */
function connectToVoice(guild) {
  try {
    const ayarlar = loadAyarlar();
    const guildSettings = ayarlar[guild.id];
    if (!guildSettings || !guildSettings.sesTeyitKanal) return;

    const voiceChannel = guild.channels.cache.get(guildSettings.sesTeyitKanal);
    if (!voiceChannel) return;

    if (getVoiceConnection(guild.id)) return;

    const connection = joinVoiceChannel({
      channelId: voiceChannel.id,
      guildId: guild.id,
      adapterCreator: guild.voiceAdapterCreator,
      selfDeaf: false,
      selfMute: true,
    });

    connection.on(VoiceConnectionStatus.Disconnected, async () => {
      try {
        await Promise.race([
          entersState(connection, VoiceConnectionStatus.Signalling, 5_000),
          entersState(connection, VoiceConnectionStatus.Connecting, 5_000),
        ]);
      } catch {
        connection.destroy();
        setTimeout(() => connectToVoice(guild), 10_000);
      }
    });

    connection.on(VoiceConnectionStatus.Destroyed, () => {
      setTimeout(() => connectToVoice(guild), 10_000);
    });

    connection.on('error', (err) => {
      console.error(`[${guild.name}] Ses bağlantısı hatası:`, err);
      connection.destroy();
      setTimeout(() => connectToVoice(guild), 10_000);
    });

  } catch (error) {
    console.error(`[${guild.name}] Ses kanalına bağlanma hatası:`, error);
    setTimeout(() => connectToVoice(guild), 10_000);
  }
}

/* ───────────────────────────────
   Bot hazır olduğunda
─────────────────────────────── */
client.once('ready', async () => {
  console.log(`✅ Bot aktif: ${client.user.tag}`);

  updateStatus();
  setInterval(updateStatus, 60_000);

  const rest = new REST({ version: '10' }).setToken(process.env.DISCORD_TOKEN);

  try {
    console.log('Slash komutlar kaydediliyor...');
    await new Promise(r => setTimeout(r, 2000));
    for (const guild of client.guilds.cache.values()) {
      await rest.put(
        Routes.applicationGuildCommands(client.user.id, guild.id),
        { body: commands }
      );
    }
    console.log('✅ Slash komutlar başarıyla kaydedildi!');
  } catch (error) {
    console.error('Komut kaydı hatası:', error);
  }

  for (const guild of client.guilds.cache.values()) connectToVoice(guild);

  setInterval(() => {
    for (const guild of client.guilds.cache.values()) {
      if (!getVoiceConnection(guild.id)) connectToVoice(guild);
    }
  }, 60_000);
});

/* ───────────────────────────────
   Interaction işlemleri
─────────────────────────────── */
client.on('interactionCreate', async (interaction) => {
  // ModalSubmit dahil
  if (
    !interaction.isChatInputCommand() &&
    !interaction.isStringSelectMenu() &&
    !interaction.isButton() &&
    !interaction.isModalSubmit()
  ) return;

  // Slash komutlar
  if (interaction.isChatInputCommand()) {
    const command = client.commands.get(interaction.commandName);
    if (!command) return;

    try {
      await command.execute(interaction, client);
    } catch (error) {
      console.error('Komut hatası:', error);
      const msg = { content: '⚠️ Komut çalıştırılırken bir hata oluştu!', ephemeral: true };
      if (interaction.replied || interaction.deferred) await interaction.followUp(msg);
      else await interaction.reply(msg);
    }
    return;
  }

  // ✅ Select / Button / Modal => tüm komutların handleInteraction'ını dene
  if (interaction.isStringSelectMenu() || interaction.isButton() || interaction.isModalSubmit()) {
    const handlers = [...client.commands.values()].filter(
      (c) => c && typeof c.handleInteraction === 'function'
    );

    for (const cmd of handlers) {
      try {
        await cmd.handleInteraction(interaction, client);

        // biri cevap verdiyse dur
        if (interaction.replied || interaction.deferred) return;
      } catch (error) {
        console.error('İnteraction hatası:', error);

        if (!interaction.replied && !interaction.deferred) {
          await interaction.reply({ content: '⚠️ İşlem sırasında hata oluştu.', ephemeral: true }).catch(() => {});
        }
        return;
      }
    }
  }
});

/* ───────────────────────────────
   Token kontrolü
─────────────────────────────── */
if (!process.env.DISCORD_TOKEN) {
  console.error('❌ DISCORD_TOKEN bulunamadı! Lütfen .env dosyasına ekleyin.');
  process.exit(1);
}

client.login(process.env.DISCORD_TOKEN);
